
//
// Trust Design LLC : SECS/HSMS Communication library
//
// (c) Copyright Trust Design LLC.  2010-2019.  All rights reserved.
//


// =============================================================================
//
// CSIo : Test and sample program
//
//  Construct a message in AP and simply send and receive SECS messages.
//
//  (Note) This sample uses MSVCRT.
//
//
// Starting method
//
//   CSIo {h|e}
//   ~~~~~~~~~~
//   h    : Refer to [HOST]  section of Sample.ini to determine operation
//   e    : Refer to [EQUIP] section of Sample.ini to determine operation
//
//
//   Normally, "CSIo h" and "CSIo e" both operate on same machine or
//   different machines to communicate with each other. The communication
//   partner may be BasicIo.exe or Callbackio.exe.
//
//   After startup, refer to the menu display, enter request code etc. and the
//   function is executed.
// 
//   When host starts up, it waits for a connection from equipment.
//   When device side is started, connection request and Select request are
//   automatically issued to host side.
//   On host side, repeat '1:Recv' processing, and confirm that display is
//   'STATUS=951: No data'. After that, exchange any message. On receiving side,
//   do not forget to reap received data until 'STATUS=951'.
//
//   The transaction ID used when sending secondary message uses that of primary
//   message received immediately before.
//
//   This sample omits some of abnormal processing.
//
//   "SECS_MODE" defined below is defined to determine whether or not connection
//   processing to Passive side in case of Active connection is performed when
//   using TDS._CommXxxx() as processing function.
//
//   "FUNC_TYPE" is defined to select type of processing function to be used.
//   If you use TDS._UDrvXxxx(), there is no need to perform connection
//   processing for HSMS-SS Active connection, so there is no need to set
//   "SECS_MODE" in principle.
//   Since TDS switches SECS-1 connection or HSMS-SS connection in setting file
//   (.ini), user AP does not need to know connection method (SECS-1or HSMS and
//   Passive/Active at HSMS) as long as TDS._UDrvXxxx() is used as processing
//   function.
//
//   By changing values of "MSSG_USE_FILE" and "MSSG_DISP_TYPE" defined below,
//   you can change output format of SECS message displayed by this AP.
//   By changing value of "MSSG_USE_NEXTL", it is possible to change whether to
//   acquire field values as data or display list format when analyzing message
//   contents.
//   Similarly, if "USE_CALLBACK" is set to 1, use Callback function to perform
//   receiving processing, and automatically display received message and output
//   secondary message.
//   When using Callback function, keep in mind that input prompting message may
//   be disturbed.
//
//
// =============================================================================
//
// CSIo : eXg y TvEvO
//
//  `oŃbZ[W\zAP SECS bZ[W̑MsB
//
//  () {Tv́AMSVCRT gpB
//
//
// N@
//
//   CSIo {h|e}
//   ~~~~~~~~~~
//   h    : Sample.ini  [HOST]  ZNVQƂ肷
//   e    : Sample.ini  [EQUIP]      :              :
//
//
//   ʏACSIo h y CSIo e ̗A}VA͈قȂ}V
//   삳āAݒʐMsBʐḾABasicIo.exeACallbackio.exe ł
//   悢B
//
//   NAj\QƂAvR[h͂A@\sB
// 
//   zXg͋NƁAu̐ڑ҂BuNƁAI
//   ɃzXgɐڑvASelect v𔭍sBzXgł́A1:Recv 
//   JԂsA\uSTATUS=951 : No datavƂȂ̂mFB̌A
//   Cӂ̃bZ[ŴƂsBMł́AMf[^̊u
//   STATUS=951vɂȂ܂ōsƂYʂƁB
//
//   QbZ[W𑗐Mۂ̃gUNVhćAOɎMP
//   bZ[Ŵ̂gpB
//
//   {Tv́Aُ펞ꕔȗĂB
//
//   ȉŒ` "SECS_MODE" ́A֐Ƃ TDS._CommXxxx() gp
//   ꍇ Active ڑ̏ꍇ Passive ւ̐ڑ̎s̗L𔻒f
//   ߂ɒ`B
//   "FUNC_TYPE" ́Agp鏈֐̎ʂI邽߂ɒ`B
//   TDS._UDrvXxxx() gpꍇ́AHSMS-SS Active ڑ̏ꍇ̐ڑ
//   sKvȂ̂ŁA{ł "SECS_MODE" ̐ݒKvȂB
//   TDS ́ASECS-1 ڑ or HSMS-SS ڑ̐ؑւ́Aݒt@C (.ini) ɂčs
//   ߁A֐Ƃ TDS._UDrvXxxx() gṕA[U`oł́A
//   ڑiSECS-1/HSMS-SS  HSMS-SS Passive/ActivejmKv
//   ȂB
//
//   ȉŒ` "MSSG_USE_FILE"A"MSSG_DISP_TYPE" ̒lύXƁA{`o
//   ɂĕ\ SECS bZ[W̏o͌`ύXłB
//   "MSSG_USE_NEXTL" ̒lύXƁAbZ[We͂ۂɁAڒl
//   f[^ƂĎ擾邩A\Xg`Ŏ擾邩ύXłB
//   l "USE_CALLBACK"  =1 ƂƁAMs Callback ֐gp
//   AMbZ[W̕\AQbZ[W̏o͂ōsB
//   Callback ֐gpꍇA͑ipbZ[Wɗꂪ鎖ɗ
//   邱ƁB
//
// =============================================================================

using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.Threading;

using TDCSL;


namespace TDCSSTest{

unsafe class CSIo{
[DllImport( "msvcrt", CallingConvention=CallingConvention.Cdecl)]
public static extern int printf(string form);
[DllImport( "msvcrt", CallingConvention=CallingConvention.Cdecl)]
public static extern int printf(string form, __arglist);
[DllImport( "msvcrt", CallingConvention=CallingConvention.Cdecl)]
public static extern int sprintf(byte[] str, string form, __arglist);
[DllImport( "msvcrt", CallingConvention=CallingConvention.Cdecl)]
public static extern int scanf(string form, __arglist);


// -----------------------------------------------------------------------------

private const int	EBADF		=9;
private const int	EBUSY		=16;
private const int	ENOMEM		=12;
private const int	ENODEV		=19;
private const int	E2BIG		=7;

private const int	E_NOTCONNECT	=999;
private const int	E_DESELECT	=998;
private const int	E_REJECT	=997;
private const int	E_SELECT	=992;
private const int	E_CONNECT	=991;
private const int	E_RETRYOVER	=989;
private const int	E_T8TIMEDOUT	=988;
private const int	E_T7TIMEDOUT	=987;
private const int	E_T6TIMEDOUT	=986;
private const int	E_T5TIMEDOUT	=985;
private const int	E_T4TIMEDOUT	=984;
private const int	E_T3TIMEDOUT	=983;
private const int	E_T2TIMEDOUT	=982;
private const int	E_T1TIMEDOUT	=981;
private const int	E_ILLBLOCK	=980;
private const int	E_NODATA	=951;


// -----------------------------------------------------------------------------

private static int	SECS_MODE	=1;	// SECS/HSMS mode
						// 0    : SECS-1
						// 1    : HSMS
private static int	FUNC_TYPE	=0;	// Type of function used
						// ʐMɎgp֐̎
						// 0    : TDS._CommXxxxx
						// 1    : TDS._UDrvXxxxx
private static uint	UDRV_MASK	=0x8383ffff;
						// Mask value of UDrvOpen()
						// 0          : Set =0x49
						// 0x8383ffff : All event

private static int	USE_CALLBACK	=1;	// Use of Callback function
						// 0    : Not use
						// 1    : Use

private static int	MSSG_USE_FILE	=0x80;	// Message definition file
						// 0x00 : Not use
						// 0x80 : Use to display item
						//	  names
						//	  gpčږ\
private static int	MSSG_DISP_TYPE	=0x20;	// SECS Message display format
						// 0x00 : TDS Format
						// 0x20 : SML Format
private static int	MSSG_USE_NEXTL	=1;	// Use MssgNextL() or not
						// 0    : Not use
						// 1    : Use

private static string	PARAMFILE	="Sample.ini";


// -----------------------------------------------------------------------------

private static int	Fd    =0;	// Communication identifier
private static int	Md    =0;	// Message analysis identifier
private static int	OType =0;	// Operation type  (0:Host  1:Equipment)
private static int	Break =0;	// End instruction to thread
					// Xbhւ̏Iw

private static int	cnt249=0;
private static int	cnt250=0;
private static int	cnt611=0;

private static System.Object	Cs1=new System.Object();


// =============================================================================
// Common functions ============================================================


// -----------------------------------------------------------------------------
// Display SECS messages on standard output ------------------------------------
// SECS bZ[WWo͂ɕ\ -----------------------------------------

private static void
DispSECSMssg(			//
int		tp,		// i  : Message type
				//	=0 : Transmission result
				//	 1 : Received message
				//	 2 : Send message
byte[]		hd,		// i  : SECS Message Header
int		did,		// i  : SECS Message DeiceID
int		sf,		// i  : SECS Message SF-Cide
uint		xid,		// i  : SECS Message TransactionID
byte[]		msg,		// i  : SECS Message strage area
				//	     (Header not included)
int		len)		// i  : SECS Message byte length
{
  string[]	ctp={"SRES","RECV","SEND"};
  string	mname,sitem;
  byte  []	itm=new byte  [64];
  short []	vi2=new short [ 1];
  ushort[]	vu2=new ushort[ 1];
  int   []	vi4=new int   [ 1];
  uint  []	vu4=new uint  [ 1];
  long  []	vi8=new long  [ 1];
  ulong []	vu8=new ulong [ 1];
  float []	vf4=new float [ 1];
  double[]	vf8=new double[ 1];
  string	str,rbit,wbit;
  byte  []	sfcode=new byte[16];
  int		rtn,md=0,dp=MSSG_USE_FILE|MSSG_DISP_TYPE,fm,form,sz,noi,la,i;
	// [Note] Refer to [Note] described in DispSECSMssg () in SubFuncsion.h
	// [] SubFuncsion.h  DispSECSMssg() ɋLq [] QƂ邱

  rbit=" ";	if((did&0x8000)!=0)	rbit="R";
  wbit=" ";	if((sf &0x8000)!=0)	wbit="W";
  sprintf(sfcode,"S%dF%d",__arglist((sf&0x7f00)/0x0100,sf&0xff));
  printf("[%s]  Dev=0x%04x  %-8s  %s%s  XId=0x%04x  Len=%4d\n"
		,__arglist(ctp[tp],did&0x7fff,sfcode,rbit,wbit,xid,len));

  if((MSSG_USE_FILE&0x80)!=0)	fm =0x8000;	// Use message definition file
  else				fm =0x0000;	// Not use

  if(len>0){
    if(tp==1){	dp|=0x3000;	fm|=0x3000;}	// In case of receiving
    else{	dp|=0x2000;	fm|=0x2000;}	// In case of sending

    if((md=TDS._MssgFind(fm,msg,len,Fd,hd,out mname))	> 0){
      if(mname!="") printf("[%s]\n",__arglist(mname));	// Message name

      for(la=0;;){
	if(MSSG_USE_NEXTL==0){	// Get item value
	  fixed(byte* pm=itm){
	    rtn=TDS._MssgNext(md,0,msg,out form,out sz,out noi,pm,64,out sitem);
	    if(rtn					< 0)	break;
	    printf("%03o:%d*%2d:",__arglist(form,sz,noi));
	    switch(form){	// Display field value
				// The second and subsequent numbers are omitted
				// l̂QԖڈȍ~͏ȗ
	      case 0x00: Console.WriteLine("L[{0}]"	    ,noi);	   break;
	      case 0x08: Console.WriteLine("B[{0}]={1,0:x2}",noi,itm[0]);  break;
	      case 0x09: Console.WriteLine("T[{0}]={1,0}"   ,noi,itm[0]);  break;
	      case 0x10: Console.WriteLine("A[{0}]={1}"	    ,noi,sitem);   break;
	      case 0x11: Console.WriteLine("J[{0}]={1}"	    ,noi,sitem);   break;
	      case 0x12: Console.WriteLine("K[{0}]={1}"	    ,noi,sitem);   break;
	      case 0x18: Buffer.BlockCopy(itm,0,vi8,0,sz);
			 Console.WriteLine("I8[{0}]={1}"    ,noi,vi8[0]);  break;
	      case 0x19: Console.WriteLine("I1[{0}]={1}"    ,noi,itm[0]);  break;
	      case 0x1a: Buffer.BlockCopy(itm,0,vi2,0,sz);
			 Console.WriteLine("I2[{0}]={1}"    ,noi,vi2[0]);  break;
	      case 0x1c: Buffer.BlockCopy(itm,0,vi4,0,sz);
			 Console.WriteLine("I4[{0}]={1}"    ,noi,vi4[0]);  break;
	      case 0x20: Buffer.BlockCopy(itm,0,vf8,0,sz);
			 Console.WriteLine("F8[{0}]={1}"    ,noi,vf8[0]);  break;
	      case 0x24: Buffer.BlockCopy(itm,0,vf4,0,sz);
			 Console.WriteLine("F4[{0}]={1}"    ,noi,vf4[0]);  break;
	      case 0x28: Buffer.BlockCopy(itm,0,vu8,0,sz);
			 Console.WriteLine("U8[{0}]={1}"    ,noi,vu8[0]);  break;
	      case 0x29: Console.WriteLine("U1[{0}]={1}"    ,noi,itm[0]);  break;
	      case 0x2a: Buffer.BlockCopy(itm,0,vu2,0,sz);
			 Console.WriteLine("U2[{0}]={1}"    ,noi,vu2[0]);  break;
	      case 0x2c: Buffer.BlockCopy(itm,0,vu4,0,sz);
			 Console.WriteLine("U4[{0}]={1}"    ,noi,vu4[0]);  break;
	      default:	 Console.WriteLine("");				   break;
	  } }

	}else{					// Get in list format
						// Xg\`Ŏ擾
	  rtn=TDS._MssgNextL(md,dp,msg,out form,out noi,out str);
	  if(rtn					< 0)	break;
	  if((dp&0x70)>0x10){			// In case of SML
	    for(i=la;i>rtn;i--){
	      printf("%*s>\n"	,__arglist(i*2,""));	// Display '>'
	  } }
	  la=rtn;				// Save current hierarchy
						// ݂̊Kwۑ
	  printf("  %s\n",__arglist(str));	// Display acquired field value
						// 擾ڒl\
	  if((dp&0x70)>0x10 && form==000 && noi==0){
	    printf("%*s>\n"	,__arglist((la+1)*2,""));
						// '>' Processing for L0
      } } }					// L0 ̏ꍇ '>' 
      if(MSSG_USE_NEXTL!=0){
	if((dp&0x70)>0x10){
	   for(i=la;i>0;i--){
	     printf("%*s>\n"	,__arglist(i*2,""));	// Show remaining '>'
      } } }						// c '>' \
      TDS._MssgExit(md,0,msg);
  } }
}



// -----------------------------------------------------------------------------
// Display sent and received data ----------------------------------------------

private static int
DispData(
int		tp,		// i  : Message type
				//	=0 : Transmission result
				//	 1 : Received message
				//	 2 : Send message
byte[]		hd,		// i  : SECS Message Header
int		did,		// i  : SECS Message Device ID
int		sf,		// i  : SECS Message SF-Code
uint		xid,		// i  : SECS Message Transaction ID
byte[]		msg,		// i  : SECS Message Body
int		len,		// i  : Byte length of 'msg'
int		rtn)		// i  : I/O return value
{
  int		ok=0,stat;

  lock(Cs1){			// See notes in SubFuncsion.h
				// SubFuncsion.h ̒LQƂ邱
  if(rtn<0){							 ok=1;
    if((rtn<(-E_NOTCONNECT) || (-E_ILLBLOCK)<rtn) && rtn!=(-E_NODATA)){
      printf("ERROR  [%d]"			,__arglist(rtn));
      if      (rtn==(-ENODEV)){
	printf(" : No such device ID\n");	len=ok=0;
      }else if(rtn==(-E2BIG )){
	printf(" : Data size to large\n");	len=ok=0;
      }else			printf("\n");
    }else{
      if(FUNC_TYPE==0)	stat=TDS._CommStatus(Fd,0);
      else		stat=TDS._UDrvStatus(Fd,0);
      printf("STATUS = %d,%d : "		,__arglist(-rtn,stat));
      switch(-rtn){
	case E_NODATA:		printf("No data\n");			break;
	case E_ILLBLOCK:	printf("Illegal block#\n");		break;
	case E_T1TIMEDOUT:	printf("T1 Timeout occur\n");		break;
	case E_T2TIMEDOUT:	printf("T2 Timeout occur\n");		break;
	case E_T3TIMEDOUT:	printf("T3 Timeout occur\n");		break;
	case E_T4TIMEDOUT:	printf("T4 Timeout occur\n");		break;
	case E_T5TIMEDOUT:	printf("T5 Timeout occur\n");		break;
	case E_T6TIMEDOUT:	printf("T6 Timeout occur\n");		break;
	case E_T7TIMEDOUT:	printf("T7 Timeout occur\n");		break;
	case E_T8TIMEDOUT:	printf("T8 Timeout occur\n");		break;
	case E_RETRYOVER:	printf("Retry over\n");			break;
	case E_CONNECT:		printf("Connected\n");			break;
	case E_SELECT:		printf("Selected   (0x%04x)\n"
						,__arglist(did));	break;
	case E_REJECT:		printf("Rejected XId=0x%04x\n"
						,__arglist(xid));	break;
	case E_DESELECT:	printf("Deselected (0x%04x)\n"
						,__arglist(did));	break;
	case E_NOTCONNECT:	printf("Not connected\n");		break;
	default:		printf("\n");				break;
  } } }

  if(ok==0) DispSECSMssg(tp,hd,did,sf,xid,msg,len);
  }

  return(0);
}



// =============================================================================
// S1F1 message construction and sending ---------------------------------------

private static int
SendS1F1()
{
  byte[]	hd=new byte[12],msg=new byte[16];
  int		rtn,sf=0x8101,len=0;

		// See (Note) in SendS1F1() of SubFunction.h.
  		// SubFunction.h  SendS1F1()  () QƂ邱ƁB
  if(FUNC_TYPE==0)   rtn=TDS._CommSend(Fd,0x0000,0,sf,0,msg,len,hd);
  else		     rtn=TDS._UDrvSend(Fd,0x0000,0,sf,0,msg,len,hd);
  DispData(2,hd,0x00,sf,(uint)rtn,msg,len,rtn);

  return(rtn);
}


// -----------------------------------------------------------------------------
// S1F2 message (Host) construction and sending --------------------------------

private static int
SendS1F2H(
int		did,
uint		xid)
{
  byte[]	hd=new byte[12],msg=new byte[256];
  byte[]	vb=new byte[32];
  int		rtn,md,sf=0x0102,len;
  
  if((MSSG_USE_FILE&0x80)==0){			// Do not use message definition
    md=		TDS._MssgInit   (   0,msg, 256,Fd);		// S1F2
		TDS._MssgBuild  (md,0,msg,0x00,  0,vb);		// L0
    len=	TDS._MssgEnd    (md,0,msg);

  }else{					// Use message definition
		TDS._MDMssgInit (Md,0,msg, 256,"S1F2_H");
    len=	TDS._MDMssgEnd  (Md,0,msg);
  }

  if(len<0)	     rtn=len;
  else{
    if(FUNC_TYPE==0) rtn=TDS._CommSend (Fd,0x0000,did,sf,xid,msg,len,hd);
    else	     rtn=TDS._UDrvSend (Fd,0x0000,did,sf,xid,msg,len,hd);
  }
  DispData(2,hd,did,sf,(uint)rtn,msg,len,rtn);

  return(rtn);
}


// -----------------------------------------------------------------------------
// S1F2 message (Equipment) construction and sending ---------------------------

private static int
SendS1F2E(
int		did,
uint		xid)
{
  byte[]	hd=new byte[12],msg=new byte[256];
  byte[]	vb=new byte[32];
  int		rtn,md,sf=0x0102,len;
  
  if((MSSG_USE_FILE&0x80)==0){			// Do not use message definition
    md=		TDS._MssgInit   (   0,msg, 256,Fd);		// S1F2
		TDS._MssgBuild  (md,0,msg,0x00, 2,vb);		// L2
		TDS._MssgBuild  (md,0,msg,0x10, 6,"EQUIP1");	//  MDLN
		TDS._MssgBuild  (md,0,msg,0x10, 6,"01.000");	//  SOFTREV
    len=	TDS._MssgEnd    (md,0,msg);

  }else{					// Use message definition
		TDS._MDMssgInit (Md,0,msg, 256,"S1F2_E");
    len=	TDS._MDMssgEnd  (Md,0,msg);
  }

  if(len<0)	     rtn=len;
  else{
    if(FUNC_TYPE==0) rtn=TDS._CommSend (Fd,0x0000,did,sf,xid,msg,len,hd);
    else	     rtn=TDS._UDrvSend (Fd,0x0000,did,sf,xid,msg,len,hd);
  }
  DispData(2,hd,did,sf,(uint)rtn,msg,len,rtn);

  return(rtn);
}


// =============================================================================
// S2F49 message construction and sending --------------------------------------

private static int
SendS2F49()
{
  byte[]	hd=new byte[12],msg=new byte[1024];
  byte[]	vb=new byte[32];
  int[]		vi=new int[4];
  string	str,itm;
  int		rtn,md,sf=0x8200+49,no1,no2,len,i,j;

  cnt249++;	no1=(cnt249%2)+1;	no2=(cnt249%10)+1;
  str=string.Format("LOTID ({0,0:d4})",cnt249);
  
  if((MSSG_USE_FILE&0x80)==0){			// Do not use message definition
    md=		TDS._MssgInit   (   0,msg,1024,Fd);		// S2F49
  		TDS._MssgBuild  (md,0,msg,0x00,  3,vb);		// L3
    vb[0]=0;	TDS._MssgBuild  (md,0,msg,0x08,  1,vb);		//  DATAIDB
  		TDS._MssgBuild  (md,0,msg,0x10,  0,"LOAD");	//  RCMD
  		TDS._MssgBuild  (md,0,msg,0x00,  4,vb);		//  L4
    vb[0]=1;	TDS._MssgBuild  (md,0,msg,0x08,  1,vb);		//   STID
    vb[0]=0;	TDS._MssgBuild  (md,0,msg,0x08,  1,vb);		//   MTKD
  		TDS._MssgBuild  (md,0,msg,0x10, 20,str);	//   LOTID
		TDS._MssgBuild  (md,0,msg,0x00,no1,vb);		//   L[no1]
    for(i=0;i<no1;i++){
		TDS._MssgBuild  (md,0,msg,0x00,no2,vb);		//    L[no2]
      for(j=0;j<no2;j++){
		TDS._MssgBuild  (md,0,msg,0x00,  2,vb);		//     L[2]
	str=string.Format("WAFER({0,0:d4}-{1,0:d1}-{2,0:d2})",cnt249,i+1,i+2);
  		TDS._MssgBuild  (md,0,msg,0x10, 20,str);	//      WAFERID
	str=string.Format("PPID ({0,0:d4}-{1,0:d1}-{2,0:d2})",cnt249,i+1,i+2);
  		TDS._MssgBuild  (md,0,msg,0x10, 16,str);	//      PPID
    } }
    len=	TDS._MssgEnd    (md,0,msg);

  }else{					// Use message definition
		TDS._MDMssgInit (Md,0,msg,1024,"S2F49");
  		TDS._MDMssgBuild(Md,0,msg,"LOTID",0,str);	//   LOTID
    vi[0]=no1;	TDS._MDMssgBuild(Md,0,msg,"NOI1" ,1,vi);	//   L[no1]
    for(i=0;i<no1;i++){
      itm  =string.Format("NOI2:{0,0:d1}"		,i+1);
      vi[0]=no2;TDS._MDMssgBuild(Md,0,msg,itm   ,1,vi);		//    L[no2]
      for(j=0;j<no2;j++){
	itm=string.Format("WAFERID:{0,0:d1}:{1,0:d1}"	,i+1,j+1);
	str=string.Format("WAFERID[{0,0:d4}-{1,0:d1}-{1,0:d2}]"
						,cnt249	,i+1,j+1);
  		TDS._MDMssgBuild(Md,0,msg,itm    ,0,str);	//     WAFERID
	itm=string.Format("PPID:{0,0:d1}:{1,0:d1}"	,i+1,j+1);
	str=string.Format("PPID [{0,0:d4}-{1,0:d1}-{1,0:d2}]"
						,cnt249	,i+1,j+1);
  		TDS._MDMssgBuild(Md,0,msg,itm    ,0,str);	//     PPID
    } }
    len=	TDS._MDMssgEnd  (Md,0,msg);
  }

  if(len<0)	     rtn=len;	// See (Note) in SendS1F1() of SubFunction.h
  else{				// SubFunction.h  SendS1F1()  () Q
    if(FUNC_TYPE==0) rtn=TDS._CommSend (Fd,0x0000,0,sf,  0,msg,len,hd);
    else	     rtn=TDS._UDrvSend (Fd,0x0000,0,sf,  0,msg,len,hd);
  }
  DispData(2,hd,0x00,sf,(uint)rtn,msg,len,rtn);

  return(rtn);
}


// -----------------------------------------------------------------------------
// S2F50 message construction and sending --------------------------------------

private static int
SendS2F50(
int		did,
uint		xid)
{
  byte[]	hd=new byte[12],msg=new byte[256];
  byte[]	vb=new byte[32];
  string	str;
  int		rtn,md,sf=0x0200+50,len;
  
  str=string.Format("LOTID ({0,0:d4})"	,++cnt250);

  if((MSSG_USE_FILE&0x80)==0){			// Do not use message definition
    md=		TDS._MssgInit   (   0,msg, 256,Fd);		// S2F50
		TDS._MssgBuild  (md,0,msg,0x00,  2,vb);		// L2
    vb[0]=0;	TDS._MssgBuild  (md,0,msg,0x08,  1,vb);		//  HCACK
		TDS._MssgBuild  (md,0,msg,0x00,  2,vb);		//  L2
  		TDS._MssgBuild  (md,0,msg,0x10,  5,"PODID");	//   PODID
  		TDS._MssgBuild  (md,0,msg,0x10, 20,str);	//   LOTID
    len=	TDS._MssgEnd    (md,0,msg);

  }else{					// Use message definition
		TDS._MDMssgInit (Md,0,msg, 256,"S2F50");
		TDS._MDMssgBuild(Md,0,msg,"LOTID",  0,str);	//   LOTID
    len=	TDS._MDMssgEnd  (Md,0,msg);
  }

  if(len<0)	     rtn=len;
  else{
    if(FUNC_TYPE==0) rtn=TDS._CommSend (Fd,0x0000,did,sf,xid,msg,len,hd);
    else	     rtn=TDS._UDrvSend (Fd,0x0000,did,sf,xid,msg,len,hd);
  }
  DispData(2,hd,did,sf,(uint)rtn,msg,len,rtn);

  return(rtn);
}


// =============================================================================
// S6F11 message construction and sending --------------------------------------

private static int
SendS6F11()
{
  byte[]	hd=new byte[12],msg=new byte[256];
  ushort[]	vs=new ushort[4];
  int		rtn,md,sf=0x8600+11,len;

  if((++cnt611)==65536) cnt611=0;

  if((MSSG_USE_FILE&0x80)==0){			// Do not use message definition
    md=		TDS._MssgInit   (   0,msg, 256,Fd);		// S6F11
  		TDS._MssgBuild  (md,0,msg,0x00,  3,vs);		// L3
    vs[0]=(ushort)cnt611;
		TDS._MssgBuild	(md,0,msg,0x2a,  1,vs);		//  DATAID
    vs[0]=8;	TDS._MssgBuild  (md,0,msg,0x2a,  1,vs);		//  CEID
  		TDS._MssgBuild  (md,0,msg,0x00,  3,vs);		//  L3
  		TDS._MssgBuild  (md,0,msg,0x10, 16,"DATA1");	//   DATA1
  		TDS._MssgBuild  (md,0,msg,0x10, 16,"DATA2");	//   DATA2
  		TDS._MssgBuild  (md,0,msg,0x10, 14,"YYYYMMDDhhmmss");
			// @TIME Actually I set current time, but omitted.
			// @TIME {͌ݎݒ肷̂Aȗ
    len=	TDS._MssgEnd    (md,0,msg);

  }else{					// Use message definition
		TDS._MDMssgInit (Md,0,msg, 256,"S6F11_0");
    vs[0]=(ushort)cnt611;
		TDS._MDMssgBuild(Md,0,msg,"DATAID",1,vs);	//  DATAID
    len=	TDS._MDMssgEnd  (Md,0,msg);
  }

  if(len<0)	     rtn=len;	// See (Note) in SendS1F1().
  else{				// SubFunction.h  SendS1F1()  () QƁB
    if(FUNC_TYPE==0) rtn=TDS._CommSend (Fd,0x0000,0,sf,  0,msg,len,hd);
    else	     rtn=TDS._UDrvSend (Fd,0x0000,0,sf,  0,msg,len,hd);
  }
  DispData(2,hd,0x00,sf,(uint)rtn,msg,len,rtn);

  return(rtn);
}



// -----------------------------------------------------------------------------
// S6F12 message construction and sending --------------------------------------

private static int
SendS6F12(
int		did,
uint		xid)
{
  byte[]	hd=new byte[12],msg=new byte[256];
  byte[]	vb=new byte[32];
  int		rtn,md,sf=0x0600+12,len;

  if((MSSG_USE_FILE&0x80)==0){			// Do not use message definition
    md=		TDS._MssgInit	(   0,msg, 256,Fd);		// S6F12
    vb[0]=0;	TDS._MssgBuild	(md,0,msg,0x08,  1,vb);		// ACKC
    len=	TDS._MssgEnd 	(md,0,msg);

  }else{					// Use message definition
		TDS._MDMssgInit (Md,0,msg, 256,"S6F12");
    len=	TDS._MDMssgEnd  (Md,0,msg);
  }

  if(len<0)	     rtn=len;
  else{
    if(FUNC_TYPE==0) rtn=TDS._CommSend (Fd,0x0000,did,sf,xid,msg,len,hd);
    else	     rtn=TDS._UDrvSend (Fd,0x0000,did,sf,xid,msg,len,hd);
  }
  DispData(2,hd,did,sf,(uint)rtn,msg,len,rtn);

  return(rtn);
}



// =============================================================================
// Callback function ===========================================================

private static int
CBRecvProc(
int		req,		// i  : Request code to library
int		rtn,		// i  : Return code from library
int		did,		// i  : SECS Message Device ID
int		xsf,		// i  : SECS Message SF-Code
uint		xid,		// i  : SECS Message Transaction ID
byte[]		xhd,		// i  : SECS Message Header
byte[]		xmsg)		// i  : SECS Message Body
{
  byte[]	hd=new byte[12],msg=new byte[1024];
  string	rname,sname;
  int		len,sf;

  DispData(1,xhd,did,xsf,xid,xmsg,rtn,rtn);

  if      (req== 0 && rtn>=0){
    if((MSSG_USE_FILE&0x80)==0){// If you do not use message definition file,
				// check SF-Code on your own, determine 
				// necessity of sending secondary message, and
				// send it if necessary.
				// bZ[W`t@CgpȂꍇ́A
	    			// ͂ SF-Code 𒲂ׁAQbZ[W̑o
				// Kv𔻒fAKvȏꍇ́AoB
      if     (xsf==(0x8100+ 1)&&OType==0)	SendS1F2H (did,xid);
      else if(xsf==(0x8100+ 1)&&OType==1)	SendS1F2E (did,xid);
      else if(xsf==(0x8200+49))			SendS2F50 (did,xid);
      else if(xsf==(0x8600+11))			SendS6F12 (did,xid);

    }else if((xsf & 0x8000)!=0){// When using a message definition file, this
				// sample uses automatic reply function.
				// bZ[W`t@Cgpꍇ́A{T
				// vł́AԐM@\gpB
      msg=xmsg;
      if((len=TDS._MDMssgAutoRes(Md,0,xhd
		,msg,rtn,1024,out rname,out sname,out sf))	>=0){
	printf("RECV %s ..  Auto respond %s [S%dF%d]\n"
		,__arglist(rname,sname,sf/0x100,sf&0xff));
	if(FUNC_TYPE==0) rtn=TDS._CommSend(Fd,0x0000,did,sf,xid,msg,len,hd);
	else		 rtn=TDS._UDrvSend(Fd,0x0000,did,sf,xid,msg,len,hd);
	DispData(2,hd,did,sf,xid,msg,len,rtn);
      }else{
	if(len!=(-930) && len!=(-931)){
	  printf("RECV Auto response error (%d)\n",__arglist(len));
  } } } }

  return(0);
}



// -----------------------------------------------------------------------------

private static void
RecvProcThread(
object		param)
{
  byte[]	msg=new byte[1024],hd=new byte[12];
  uint		xid;
  int		rtn,req,did,sf;

  for(;Break==0;){
    req=0;
    if(FUNC_TYPE==0){
      rtn=TDS._CommRecv(Fd,0,out did,out sf,out xid,msg,1024,hd);
    }else{
      rtn=TDS._UDrvRecv(Fd,0,out did,out sf,out xid,msg,1024,hd);
    }
    if(rtn==(-951)){
      Thread.Sleep(100);
    }else{
      if((-1000)<rtn && rtn<(-959))	req=(-rtn)-900;
      CBRecvProc(req,rtn,did,sf,xid,hd,msg);
  } }
}



// =============================================================================
// Main process ----------------------------------------------------------------

static int
Main(
string[]		argv)
{
  if(argv.Length<1 || (argv[0]!="h" && argv[0]!="e")){
    printf("Usage: CSIo {h|e}\n");

  }else{
    if(argv[0]=="h")	Host ();
    else		Equip();
  }

  return(0);
}



// =============================================================================
// Host side process -----------------------------------------------------------

private static void
Host()
{
  Thread		th;
  byte[]		hd=new byte[12],msg=new byte[1024];
  uint			xid,xids=0;
  int			did,dids=0;
  int			req,rtn,mno,sf=0;

  OType=0;		Break=0;
  if(FUNC_TYPE==0)	Fd=TDS._CommOpen(0x02,PARAMFILE,"HOST");
  else			Fd=TDS._UDrvOpen(0x02,PARAMFILE,"HOST",(int)UDRV_MASK);
  if(Fd								< 0) goto Exit;
  printf("(H) Opened (%d)\n",__arglist(Fd));
  if((MSSG_USE_FILE&0x80)!=0) Md=TDS._MDMssgInitialize(0x4000,Fd,"");

  if(USE_CALLBACK!=0){
    th=new Thread(new ParameterizedThreadStart(RecvProcThread));
    th.Start("");
  }

  for(;;){		rtn=0;
    if(USE_CALLBACK==0)	printf("Req (0:Exit 1:Recv 2:Send) : ");
    else		printf("Req (0:Exit 2:Send) : ");
    scanf ("%d",__arglist(out req));
    if      (req==0){						break;

    }else if(req==1){
      if(FUNC_TYPE==0){
	rtn=TDS._CommRecv(Fd,0,out did,out sf,out xid,msg,1024,hd);
      }else{
	rtn=TDS._UDrvRecv(Fd,0,out did,out sf,out xid,msg,1024,hd);
      }
      if(rtn	>=0){	dids=did;	xids=xid;}
      DispData(1,hd,did,sf,xid,msg,rtn,rtn);

    }else if(req==2){
      if(USE_CALLBACK==0) printf("Message(1:S1F1 2:S2F49  6:S1F2 7:S6F12) : ");
      else		  printf("Message(1:S1F1 2:S2F49) : ");
      scanf("%d",__arglist(out mno));
      switch(mno){
	case 1: rtn=SendS1F1 ();		break;
	case 2: rtn=SendS2F49();		break;
	case 6: rtn=SendS1F2H(dids,xids);	break;
	case 7: rtn=SendS6F12(dids,xids);	break;
    } }
    if(rtn<(-999) || ((-900)<rtn && rtn<0)){
      printf("(H) I/O Error (%d)\n"	,__arglist(rtn));
  } }

Exit:
  Break=1;
  if(Md>0)		TDS._MDMssgTerminate(Md,0);
  if(Fd>0){
    if(FUNC_TYPE==0)	TDS._CommClose(Fd,0);
    else		TDS._UDrvClose(Fd,0);
  } else	printf("(H) Error (%d)\n"	,__arglist(Fd));
}



// =============================================================================
// Equipment side process ------------------------------------------------------

private static void
Equip()
{
  Thread		th;
  byte[]		hd=new byte[12],msg=new byte[1024];
  uint			xid,xids=0;
  int			did,dids=0;
  int			req,rtn,mno,sf=0;

  OType=1;		Break=0;
  if(FUNC_TYPE==0)	Fd=TDS._CommOpen(0x02,PARAMFILE,"EQUIP");
  else			Fd=TDS._UDrvOpen(0x02,PARAMFILE,"EQUIP",(int)UDRV_MASK);
  if(Fd								< 0) goto Exit;
  printf("(E) Opened (%d)\n",__arglist(Fd));
  if((MSSG_USE_FILE&0x80)!=0) Md=TDS._MDMssgInitialize(0x4000,Fd,"");

  if(USE_CALLBACK!=0){
    th=new Thread(new ParameterizedThreadStart(RecvProcThread));
    th.Start("");
  }

  if(SECS_MODE!=0 && FUNC_TYPE==0){ // In case of HSMS and use TDS._CommXxxxx()
				    // HSMS  TDS._CommXxxxx() gp̏ꍇ
    if(TDS._CommSend(Fd,0x0100,0,0,0,msg,0,hd)			< 0) goto Exit;
    printf("(E) Connected\n");
    if(TDS._CommSend(Fd,0x0200,0,0,0,msg,0,hd)			< 0) goto Exit;
    printf("(E) Selected\n");
  }

  for(;;){		rtn=0;
    if(USE_CALLBACK==0)	printf("Req (0:Exit 1:Recv 2:Send) : ");
    else		printf("Req (0:Exit 2:Send) : ");
    scanf ("%d",__arglist(out req));
    if      (req==0){						 break;

    }else if(req==1){
      if(FUNC_TYPE==0){
	rtn=TDS._CommRecv(Fd,0,out did,out sf,out xid,msg,1024,hd);
      }else{
	rtn=TDS._UDrvRecv(Fd,0,out did,out sf,out xid,msg,1024,hd);
      }
      if(rtn	>=0){	dids=did;	xids=xid;}
      DispData(1,hd,did,sf,xid,msg,rtn,rtn);

    }else if(req==2){
      if(USE_CALLBACK==0) printf("Message(1:S1F1 2:S6F11  6:S1F2 7:S2F50) : ");
      else		  printf("Message(1:S1F1 2:S6F11) : ");
      scanf("%d",__arglist(out mno));
      switch(mno){
	case 1: rtn=SendS1F1 ();		break;
	case 2: rtn=SendS6F11();		break;
	case 6: rtn=SendS1F2E(dids,xids);	break;
	case 7: rtn=SendS2F50(dids,xids);	break;
    } }
    if(rtn<(-999) || ((-900)<rtn && rtn<0)){
      printf("(E) I/O Error (%d)\n"	,__arglist(rtn));
  } }

  if(SECS_MODE!=0){			// In case of HSMS, Shutdown process
					// HSMS ڑ̏ꍇAؒf
  // Deselect request is not performed. (Of course you may go. However SEMI
  // claims that HSMS-SS does not perform Deselect request.)
  // Deselect request ͍sȂB (sĂ悢BSEMI ł HSMS-SS 
  // āADeselect request ͍sȂȂAƂĂB)
  //if(FUNC_TYPE==0)	rtn=TDS._CommSend(Fd,0x0800,0,0,0,msg,0,hd);
  //else		rtn=TDS._UDrvSend(Fd,0x0800,0,0,0,msg,0,hd);
  //if(rtn							< 0) goto Exit;
  //printf("(E) Deselected\n");
    if(FUNC_TYPE==0)	rtn=TDS._CommSend(Fd,0x0900,0,0,0,msg,0,hd);
    else		rtn=TDS._UDrvSend(Fd,0x0900,0,0,0,msg,0,hd);
    if(rtn							< 0) goto Exit;
    printf("(E) Separated\n");
  }

Exit:
  Break=1;
  if(Md>0)		TDS._MDMssgTerminate(Md,0);
  if(Fd>0){
    if(FUNC_TYPE==0)	TDS._CommClose(Fd,0);
    else		TDS._UDrvClose(Fd,0);
  } else	printf("(E) Error (%d)\n"	,__arglist(Fd));
}

}
}
