
//
// Trust Design LLC : SECS/HSMS Communication library
//
// (c) Copyright Trust Design LLC.  2010-2023.  All rights reserved.
//


// =============================================================================
//
// BasicIo : Test and sample program
//
//   Construct a message in AP and simply send and receive SECS messages.
//
//   (Note) In this sample, when making an HSMS connection, it is assumed that
//	the HOST side executes Passive connection and the EQUIP side executes
//	Active connection.
// 
//
// Starting method
//
//   FirstBasicIo {h|e}
//   ~~~~~~~~~~~~~~~~~~
//   h    : Refer to [HOST]  section of Sample.ini to determine operation
//   e    : Refer to [EQUIP] section of Sample.ini to determine operation
//
//   Normally, "FirstBasicIo h" and "FirstBasicIo e" both operate on same
//   machine or different machines to communicate with each other.
//
//   After startup, it refers to the menu display, inputs request codes (0,1,2).
//   And executes functions such as receiving a message from the other
//   side and sending a message to the other side.
//
//   When host starts up, it waits for a connection from equipment.
//   When equipment side is started, connection request and Select request are
//   automatically issued to host side. On host side, repeat "1:Recv" processing,
//   and confirm that display is "STATUS = 951: No data".  After that, exchange
//   any message. On receiving side, do not forget to reap received data until
//   "STATUS = 951".
//
//   This sample omits some of abnormal processing.
//
//   This sample operates under the following conditions.
//    a. for Windows
//    b. use _TDSUDrvXxxxx()
//    c. Do not use message definition file
//    d. The connection state is synchronized with the communication control
//       thread
//    e. SECS header is accessed in internal ByteOrder state
//    f. Use _TDSMssgNextL() for the communication trace displayed on the screen
//       and output in TDL format
//    g. Use the transaction ID of previously received primary message when
//       sending secondary message.
//
//
// =============================================================================
//
// FirstBasicIo : eXg y TvEvO
//
//  `oŃbZ[W\zAP SECS bZ[W̑MsB
// 
//  "Sample0/BasicIo.c" ̓eƒPɂ́B
//  ̗ꓙ "Sample0/BasicIo.c" ̖`̃RgQƂ邱ƁB
// 
//   () {Tvł HSMS ڑꍇAHOST  Passive ڑAEQUIP 
//	Active ڑsƉ肵ĂB
//
//
// N@
//
//   FirstBasicIo {h|e}
//   ~~~~~~~~~~~~~~~~~~
//   h      : Sample.ini  [HOST]  ZNVQƂ肷
//   e      : Sample.ini  [EQUIP]      :              :
//
//   ʏAFirstBasicIo h y FirstBasicIo e ̗A}VA
//   قȂ}Vœ삳āAݒʐMsB
//
//   NAj\QƂAvR[h (0,1,2) ͂A葤
//   bZ[WMA葤փbZ[W𑗏o铙̋@\sB
// 
//   zXg͋NƁAu̐ڑ҂BuNƁAI
//   ɃzXgɐڑvASelect v𔭍sBzXgł́A1:Recv 
//   JԂsA\uSTATUS=951 : No datavƂȂ̂mFB̌A
//   Cӂ̃bZ[ŴƂsBMł́AMf[^̊u
//   STATUS=951vɂȂ܂ōsƂYʂƁB
//
//   {Tv́Aُ펞ꕔȗĂB
//
//   {Tv́Aȉ̏œ삷B
//   a. Windows p
//   b. _TDSUDrvXxxxx() gp
//   c. bZ[W`t@CgpȂ
//   d. ڑԂ͒ʐMXbhƓ
//   e. SECS wb_́A ByteOrder ԂŃANZX
//   f. ʕ\ʐMg[Xɂ _TDSMssgNextL() gp TDL `ŏo͂
//   g. QbZ[W𑗐Mۂ̃gUNVhćAOɎMP
//      bZ[Ŵ̂gpB
//
// =============================================================================

#include	"TDS.h"


#define		PO		fprintf(stdout
#define		PE		fprintf(stderr

#define		UDRV_MASK  	0x8383ffff	// Mask value for UDrvOpen()
						// 0          : Set =0x49
						// 0x8383ffff : All event
#define		PARAMFILE	"Sample.ini"

static void	Host();
static void	Equip();



// =============================================================================
// Common functions ============================================================

#include	"FirstSubFunctions.h"



// =============================================================================
// Main process ----------------------------------------------------------------

int
main(
int			argc,
char			*argv[])
{
  InitializeCriticalSection(&Cs1);

  if(argv[1][0]=='h')	Host ();
  else			Equip();

  DeleteCriticalSection(&Cs1);
  exit(0);
}



// =============================================================================
// Host side process -----------------------------------------------------------

static void
Host()
{
  TDSECSHead		hd;
  char			msg[1024];
  unsigned int		xid,xids=0;
  int			did,dids=0;
  int			req,rtn,sf,msz,mno;

  msz=sizeof(msg);
  if((Fd=_TDSUDrvOpen(0x1002,PARAMFILE,"HOST",UDRV_MASK))	< 0) goto Exit;
  PO,"(H) Opened (%d)\n",Fd);

  for(;;){
    PE,"Req (0:Exit 1:Recv 2:Send) : ");
    scanf("%d",&req);
    if      (req==0){						break;

    }else if(req==1){
      rtn=_TDSUDrvRecv(Fd,0,&did,&sf,&xid,msg,msz,&hd);
      if(rtn	  >=0){	dids=did;	xids=xid;}
      DispData(1,&hd,did,sf,xid,msg,rtn,rtn);

    }else if(req==2){
      PE,"Message(1:S1F1 2:S1F21 3:S2F49   6:S1F2 7:S1F22 8:S6F12) : ");
      scanf("%d",&mno);	rtn=0;
      switch(mno){
	case 1: rtn=SendS1F1 ();			break;
	case 2: rtn=SendS1F21();			break;
	case 3: rtn=SendS2F49();			break;
	case 6: rtn=SendS1F2H(dids,xids);		break;
	case 7: rtn=SendS1F22(dids,xids);		break;
	case 8: rtn=SendS6F12(dids,xids);		break;
    } }

    if(rtn<(-999) || ((-900)<rtn && rtn<0)){
      PE,"(H) I/O Error (%d)\n",	rtn);
  } }

Exit:
  if(Fd> 0)	_TDSUDrvClose(Fd,0);
  else		PE,"(H) Error (%d)\n",Fd);
}



// =============================================================================
// Equipment side process ------------------------------------------------------

static void
Equip()
{
  TDSECSHead		hd;
  char			msg[1024];
  unsigned int		xid,xids=0;
  int			did,dids=0;
  int			req,rtn,sf,msz,mno;

  msz=sizeof(msg);
  if((Fd=_TDSUDrvOpen(0x1002,PARAMFILE,"EQUIP",UDRV_MASK))	< 0) goto Exit;
  PO,"(E) Opened (%d)\n",Fd);

  for(;;){		rtn=0;
    PE,"Req (0:Exit 1:Recv 2:Send) : ");
    scanf("%d",&req);
    if      (req==0){						 break;

    }else if(req==1){
      rtn=_TDSUDrvRecv(Fd,0,&did,&sf,&xid,msg,msz,&hd);
      if(rtn	>=0){	dids=did; xids=xid;}
      DispData(1,&hd,did,sf,xid,msg,rtn,rtn);

    }else if(req==2){
      PE,"Message(1:S1F1 2:S1F21 3:S6F11   6:S1F2 7:S1F22 8:S2F50) : ");
      scanf("%d",&mno);	rtn=0;
      switch(mno){
	case 1: rtn=SendS1F1 ();			break;
	case 2: rtn=SendS1F21();			break;
	case 3: rtn=SendS6F11();			break;
	case 6: rtn=SendS1F2E(dids,xids);		break;
	case 7: rtn=SendS1F22(dids,xids);		break;
	case 8: rtn=SendS2F50(dids,xids);		break;
    } }

    if(rtn<(-999) || ((-900)<rtn && rtn<0)){
      PE,"(E) I/O Error (%d)\n",	rtn);
  } }

Exit:
  if(Fd> 0)	_TDSUDrvClose(Fd,0);
  else		PE,"(E) Error (%d)\n",Fd);
}
