
//
// Trust Design LLC : SECS/HSMS Communication library
//
// (c) Copyright Trust Design LLC.  2010-2023.  All rights reserved.
//


// =============================================================================
//
// FirstCallbackIo : Test and sample program
//
//   Respond to received messages using Callback function.
//
//   A simpler version of "Sample0/CallbackIo.c".
//   Refer to the comments at the beginning of "Sample0/CallbackIo.c" for 
//   the flow of processing, etc.
//
//   (Note) In this sample, when making an HSMS connection, it is assumed that
//	the HOST side executes Passive connection and the EQUIP side executes
//	Active connection.
//
//
// Starting method
//
//   FirstCallbackIo {h|e}
//   ~~~~~~~~~~~~~~~~~~~~~
//   h    : Refer to [HOST]  section of Sample.ini to determine operation
//   e    : Refer to [EQUIP] section of Sample.ini to determine operation
//
//   Normally, "FirstCallbackIo h" and "FirstCallbackIo e" both operate on same
//   machine or different machines to communicate with each other.
//
//   After startup, refer to the menu display, enter request code (0,2,3).
//   And execute functions such as sending a message to the other side.
//
//   Continuous sending and receiving is performed at a fixed interval
//   (specified by SEND_WAIT) according to request code.
//   Continuous sending and receiving is terminated by input of CTRL-C.
//   In this case, sending of primary message and sending of secondary message
//   may occur simultaneously. 
// 
//   This sample omits some of abnormal processing.
//
//   This sample operates under the following conditions.
//    a. for Windows
//    b. use _TDSUDrvXxxxx()
//    c. Do not use message definition file
//    d. The connection state is synchronized with the communication control
//       thread
//    e. SECS header is accessed in internal ByteOrder state
//    f. Use _TDSMssgNextL() for the communication trace displayed on the screen
//       and output in TDL format
//    g. Use the transaction ID of previously received primary message when
//       sending secondary message.
//
//
// =============================================================================
//
// FirstCallbackIo : eXg y TvEvO
//
//   Callback ֐gpĎMbZ[Wɑ΂鉞sB
//
//   "Sample0/CallbackIo.c" ̓eƒPɂ́B
//  ̗ꓙ "Sample0/CallbackIo.c" ̖`̃RgQƂ邱ƁB
//
//   () {Tvł HSMS ڑꍇAHOST  Passive ڑAEQUIP 
//	Active ڑsƉ肵ĂB
//
//
// N@
//
//   FirstCallbackIo {h|e}
//   ~~~~~~~~~~~~~~~~~~~~~
//   h    : Sample.ini  [HOST]  ZNVQƂ肷
//   e    : Sample.ini  [EQUIP]      :              :
//
//   ʏAFirstCallbackIo h y FirstCallbackIo e ̗A}VA
//   ͈قȂ}Vœ삳āAݒʐMsB
//
//   NAj\QƂAvR[h (0,2,3) ͂āA葤Ƀb
//   Z[W𑗏o铙̋@\sB
//   MbZ[WAyъeCxg Callback ֐̒ŏ
//   B
// 
//   vR[hɂAԊuiSEND_WAIT ɂĎwjł̘AMsB
//   AḾACTRL-C ̓͂ɂIB
//   ̏ꍇAPbZ[W̑MƁAQbZ[W̑Mɔ
//   \B
// 
//   {Tv́Aُ펞ꕔȗĂB
//
//   {Tv́Aȉ̏œ삷B
//   a. Windows p
//   b. _TDSUDrvXxxxx() gp
//   c. bZ[W`t@CgpȂ
//   d. ڑԂ͒ʐMXbhƓ
//   e. SECS wb_́A ByteOrder ԂŃANZX
//   f. ʕ\ʐMg[Xɂ _TDSMssgNextL() gp TDL `ŏo͂
//   g. QbZ[W𑗐Mۂ̃gUNVhćAOɎMP
//      bZ[Ŵ̂gpB
//
// =============================================================================

#include	"TDS.h"


#define		PO		fprintf(stdout
#define		PE		fprintf(stderr

#define		UDRV_MASK  	0x8383ffff	// Mask value for UDrvOpen()
						// 0          : Set =0x49
						// 0x8383ffff : All event
#define		PARAMFILE	"Sample.ini"


#define		SEND_WAIT	200		// Message sending interval(ms)
						// bZ[WMԊu      (mb)
#define		SEND_NO		10		// Number of times until message
						// sending off
						// bZ[WMx݂܂ł̉
#define		SEND_SLEEP	1000		// Wait time		   (ms)
						// xݎ              (mb)
#define		ERROR_SLEEP	2000		// Wait time in error case (ms)
						// G[xݎ  (mb)



static int	Finish=0;	// End instruction to thread
				// Xbhւ̏Iw


static void	Host();
static void	Equip();



// =============================================================================
// Common functions ============================================================

#include	"FirstSubFunctions.h"



// =============================================================================
// Receive processing thread ---------------------------------------------------
// MXbh ------------------------------------------------------------

static void
RecvProcThread(
void			*par)
{
  TDSCBData		cbd;
  char			msg[1024];
  int			msz,cmd=0;

  cbd.vec=1;		cbd.msg=msg;
  msz=sizeof(msg);

  for(;Finish==0;){
    cbd.rtn=_TDSUDrvRecv(Fd,0,&cbd.devid,&cbd.sf,&cbd.xid,cbd.msg,msz,&cbd.thd);
    if(cbd.rtn	== -E_NODATA)	Sleep(100);
    else{
      if((-1000)<cbd.rtn && cbd.rtn<(-959))	cbd.req=(-cbd.rtn)-900;
      else					cbd.req=0;
      CBRecvProc(par,&cbd);
      if(cbd.rtn== -E_ACCES)	Break=1;
  } }
}



// =============================================================================
// Main process ----------------------------------------------------------------

int
main(
int			argc,
char			*argv[])
{
  signal(SIGINT,CBSignal);
  InitializeCriticalSection(&Cs1);

  if(argv[1][0]=='h')	Host ();
  else			Equip();

  DeleteCriticalSection(&Cs1);
  exit(0);
}



// =============================================================================
// Host side process -----------------------------------------------------------

static void
Host()
{
  int			req,rtn,mno,i;

  if((Fd=_TDSUDrvOpen(0x1002,PARAMFILE,"HOST",UDRV_MASK))	< 0) goto Exit;
  PO,"(H) Opened (%d)\n",Fd);	OType=0;

  _beginthread(RecvProcThread,0,0);

  for(;;){
    req=99;	rtn=0;
    PE,"Req (0:Exit 2:Send 3:Continuous Send : ");
    scanf("%d",&req);
    if      (req==0){						break;

    }else if(req==2){
      PE,"Message(1:S1F1 2:S1F21 3:S2F49) : ");
      scanf("%d",&mno);	rtn=0;
      switch(mno){
	case 1: rtn=SendS1F1 ();				break;
	case 2: rtn=SendS1F21();				break;
	case 3: rtn=SendS2F49();				break;
      }

    }else if(req==3){
      for(Break=i=0;Break==0;i++){
	Sleep(SEND_WAIT);
	if(_TDSUDrvStatus(Fd,0)	==3){
	  if((rtn=SendS2F49())	< 0){
	    PE,"(H) Send error (%d)\n"	,rtn);	Sleep(ERROR_SLEEP);
	    if(rtn		==-E_ACCES)	Break=1;
	  }
	  if((i%SEND_NO)		==0)	Sleep(SEND_SLEEP);
      } }
      rtn=0;
    }

    if(rtn<(-999) || ((-900)<rtn && rtn<0) ){
      PE,"(H) Send error (%d)\n"	,rtn);
  } }

Exit:
  Finish=1;
  if(Fd> 0)	_TDSUDrvClose(Fd,0);
  else		PE,"(H) Error (%d)\n",Fd);
}



// =============================================================================
// Equipment side process ------------------------------------------------------

static void
Equip()
{
  int			req,rtn,mno,i;

  if((Fd=_TDSUDrvOpen(0x1002,PARAMFILE,"EQUIP",UDRV_MASK))	< 0) goto Exit;

  _beginthread(RecvProcThread,0,0);

  PO,"(E) Opened (%d)\n",Fd);	OType=1;

  for(;;){
    req=99;	rtn=0;
    PE,"Req (0:Exit 2:Send 3:Continuous Send) : ");
    scanf("%d",&req);
    if      (req==0){						break;

    }else if(req==2){
      PE,"Message(1:S1F1 2:S1F21 3:S6F11) : ");
      scanf("%d",&mno);	rtn=0;
      switch(mno){
	case 1: rtn=SendS1F1 ();				break;
	case 2: rtn=SendS1F21();				break;
	case 3: rtn=SendS6F11();				break;
      }

    }else if(req==3){
      for(Break=i=0;Break==0;i++){
	Sleep(SEND_WAIT);
	if(_TDSUDrvStatus(Fd,0)	==3){
	  if((rtn=SendS6F11())	< 0){
	    PE,"(E) Send error (%d)\n"	,rtn);	Sleep(ERROR_SLEEP);
	    if(rtn		==-E_ACCES)	Break=1;
	  }
	  if((i%SEND_NO)		==0)	Sleep(SEND_SLEEP);
      } }
      rtn=0;
    }

    if(rtn<(-999) || ((-900)<rtn && rtn<0)){
      PE,"(E) Send error (%d)\n"	,rtn);
  } }

Exit:
  Finish=1;
  if(Fd> 0)	_TDSUDrvClose(Fd,0);
  else		PE,"(E) Error (%d)\n",Fd);
}
