
//
// Trust Design LLC : SECS/HSMS Communication library
//
// (c) Copyright Trust Design LLC.  2010-2024.  All rights reserved.
//


// =============================================================================
//
// CallbackIo : Test and sample program
//
//   Respond to received messages using Callback function. Also, the
//   user-defined trace output function is used to perform trace output by
//   itself.
//
//   (Note) In this sample, when making an HSMS connection, it is assumed that
//	the HOST side executes Passive connection and the EQUIP side executes
//	Active connection.
//
//
// Starting method
//
//   CallbackIo {h|e} [option [send_wait [send_sleep]]]
//   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//   h    : Refer to [HOST]  section of Sample.ini to determine operation
//   e    : Refer to [EQUIP] section of Sample.ini to determine operation
//
//   option : Options for setting the details of operation (optional)
//	      Specify as a decimal number or as a hexadecimal number with 0x or
//	      x as a prefix.
//	      The details of the specified value for each bit are shown below.
//    F       8 7654 3210
//   +----+----+----+----+
//    |       | |||| |||+-- Function to use
//    |       | |||| |||    =0: _TDSCommXxxxx()		1:_TDSUDrvXxxx()
//    |       | |||| ||+--- SECS Header Byte order
//    |       | |||| ||     =0: System order		1: Network order
//    |       | |||| |+---- Format for displaying messages in SECS list format
//    |       | |||| |	    =0: TDS Format		1: SML Format
//    |       | |||| +----- Whether to use MssgNextL() for SECS message display
//    |       | ||||	    =0: Not use			1: Use
//    |       | |||+------- Whether to use the SECS message definition file
//    |       | |||	    =0: Not use			1: Use
//    |       | ||+-------- Whether to execute SECS message reception in a
//    |       | ||	    dedicated thread
//    |       | ||	    =0: No			1: Yes
//    |       | |+--------- Whether to use the Callback function for receiving
//    |       | |	    transmission results
//    |       | |	    =0: Not use			1: Use
//    |       | +---------- Whether user-specified trace output function is used
//    |       |		    =0: Not use			1: Use
//    |       +------------ Suppress Message structure output for testing
//    |			    =0: No			1: Yes
//    +-------------------- Synchronize the connection status with communication
//			    control processing thread
//			    =0: No			1: Yes
//
//   send_wait   : Transmission interval when sending continuous messages (ms).
//   send_sleep  : Time to rest for a short time after sending a given number
//                 of times (SEND_NO) when sending continuous messages (ms).
//
//   Normally, "CallbackIo h" and "CallbackIo e" both operate on same machine or
//   different machines to communicate with each other.
//
//   After startup, refer to the menu display, enter request code (0,2,3) etc.
//   and execute functions such as sending a message to the other side.
//
//   The received message and various event information complete processing in
//   Callback function. When following "MSSG_USE_FILE" is set to 0x80, secondary
//   message corresponding to received primary message is automatically sent
//   according to contents of message definition file Sample.sml set in
//   Sample.ini.
//   When MSSG_USE_FILE=0x80 (using a message definition file), message is sent
//   out by specifying name of any message defined in message definition file.
//
//   Continuous sending and receiving is performed at a fixed interval
//   (specified by SEND_WAIT) according to request code.
//   Continuous sending and receiving is terminated by input of CTRL-C.
//   In this case, sending of primary message and sending of secondary message
//   may occur simultaneously. When creating a send message using a message
//   definition file, in this sample, only one descriptor for message analysis
//   is prepared. Therefore, it is necessary to perform lock control so that
//   message construction in sending processing and reply processing does not
//   collide. See top of SunFunctions.h for details.
// 
//   This sample omits some of abnormal processing.
//
//   The following "SECS_MODE" is defined to use _TDSCommXxxx() as a processing
//   function and to judge whether or not connection processing to Passive side
//   in case of Active connection is performed.
//   (Note) Changing this value alone does not make TDS aware of the change in
//      SECS1 or HSMS.  TDS determines SECS1 or HSMS based on the setting of
//      SECSMODE in the configuration file (.ini).  To change this value, you
//      must change the setting of SECSMODE in .ini.
//
//   If the second parameter (option) at startup is not specified, the details
//   of operation are determined by the following constants.
//
//   The following "FUNC_TYPE" is defined to select type of processing function
//   to be used.
//   When using _TDSUDrvXxxx(), there is no need to perform connection
//   processing for HSMS-SS Active connection, so there is no need to set
//   "SECS_MODE" in principle.
//   TDS switches SECS-1 connection or HSMS connection in setting file (.ini).
//   When _TDSUDrvXxxx() is used, user AP is not necessary to know connection
//   method (SECS-1 or HSMS and  Passive or Active).
//   However, when using _TDSUDrvXxxx(), Callback processing needs to be
//   performed by itself.
//
//   By changing values of "MSSG_USE_FILE" and "MSSG_DISP_TYPE" defined below,
//   you can change output format of SECS message displayed by this AP.
//   By changing value of "MSSG_USE_NEXTL", it is possible to change whether to
//   acquire field values as data or display list format when analyzing message
//   contents.
//   Similarly, when "USE_RECVTHREAD" is set to =1, a thread for performing
//   receiving process is activated, and display of received message and output
//   of the secondary message are automatically performed in the thread.
//   When using the receiving thread, keep in mind that prompt message may be
//   distorted.
//   Changing value of "USE_SEND_CB" can change use of Callback for Sending
//   result.
//   Changing value of "USE_USER_TRACE" can change use of Callback for User
//   Trace output.
//   If "HEADER_BO" is set to 1, Byte Order of SECS message header obtained at
//   the time of Receive and Send can be obtained as the original network Byte
//   Order of the SECS message.
//
//
// =============================================================================
//
// CallbackIo : eXg y TvEvO
//
//   Callback ֐gpĎMbZ[Wɑ΂鉞sB܂A[U
//   `g[Xo͊֐gpAg[Xo͂͂ōsB
//
//   () {Tvł HSMS ڑꍇAHOST  Passive ڑAEQUIP 
//	Active ڑsƉ肵ĂB
//
//
// N@
//
//   CallbackIo {h|e} [option [send_wait [send_sleep]]]
//   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//   h    : Sample.ini  [HOST]  ZNVQƂ肷
//   e    : Sample.ini  [EQUIP]      :              :
//
//   option : ̏ڍׂݒ肷IvV (ȗ\)
//	      10i  ړƂ 0x  x t^ 16i
//	      Ŏw肷B
//	      ȉɎwl̃rbg̏ڍׂȉɎB
//    F       8 7654 3210
//   +----+----+----+----+
//    |	      | |||| |||+-- gp֐
//    |	      | |||| |||    =0: _TDSCommXxxxx()		1:_TDSUDrvXxxx()
//    |	      | |||| ||+--- SECS Header Byte order
//    |	      | |||| ||	    =0: System order		1: Network order
//    |	      | |||| |+---- SECS Xg`ŃbZ[W\ꍇ̌`
//    |	      | |||| |	    =0: TDS `		1: SML `
//    |	      | |||| +----- SECS bZ[W\ MssgNextL() gp邩ۂ
//    |	      | ||||	    =0: gpȂ		1: gp
//    |	      | |||+------- SECS bZ[W`t@Cgp̗L
//    |	      | |||	    =0: gpȂ		1: gp
//    |	      | ||+-------- SECS bZ[WMpXbhŎs邩ۂ
//    |	      | ||	    =0: No			1: Yes
//    |	      | |+--------- MʎMp Callback ֐gp̗L
//    |	      | |	    =0: gpȂ		1: gp
//    |	      | +---------- [Uw̃g[Xo͊֐gp̗L
//    |	      |		    =0: gpȂ		1: gp
//    |	      +------------ ̂ Message \̏o͂}~
//    |			    =0: No			1: Yes
//    +-------------------- ڑԂʐM䕔ɓ
//			    =0: No			1: Yes
//
//   send_wait   : AbZ[WM̑MԊu (ms)
//   send_sleep  : AbZ[WMɏ̉ (SEND_NO) Mɏ
//                 Mx݂鎞 (ms)
//
//   ʏACallbackIo h y CallbackIo e ̗A}VA͈قȂ
//   }Vœ삳āAݒʐMsB
//
//   NAj\QƂAvR[h (0,2,3) ͂āA葤Ƀb
//   Z[W𑗏o铙̋@\sB
//   MbZ[WAyъeCxg Callback ֐̒ŏ
//   BL "MSSG_USE_FILE"  0x80 ɐݒ肷ƁASample.ini Őݒ肵
//   bZ[W`t@C Sample.sml ̓eɂAMPbZ[WɑΉ
//   QbZ[WoB
//   MSSG_USE_FILE=0x80ibZ[W`t@Cgpjꍇ́AbZ[W
//   `t@Cɒ`Cӂ̃bZ[Ŵ̖w肷邱ƂɂA
//   bZ[W𑗏oB
// 
//   vR[hɂAԊuiSEND_WAIT ɂĎwjł̘AMsB
//   AḾACTRL-C ̓͂ɂIB
//   ̏ꍇAPbZ[W̑MƁAQbZ[W̑Mɔ
//   \BbZ[W`t@CgpđMbZ[W쐬ꍇA
//   {Tvł́AbZ[W͗p̃fXNv^͂PpӂĂȂ
//   ŁAMAԐMł̃bZ[W\zobeBOȂ悤ɃbN
//   KvBڍׂ SunFunctions.h ̐擪QƂ邱ƁB
// 
//   {Tv́Aُ펞ꕔȗĂB
//
//   ȉŒ` "SECS_MODE" ́A֐Ƃ _TDSCommXxxx() gp
//   ꍇ Active ڑ̏ꍇ Passive ւ̐ڑ̎s̗L𔻒f
//   ߂ɒ`B
//   () ̒l̕ύX TDS  SECS1 or HSMS ̕ύXF킯ł͂ȂB 
//      TDS  SECS1 or HSMS 𔻒f̂͐ݒt@C (.ini)  SECSMODE 
//      ݒɂ̂ŁA̒lύXꍇ .ini  SECSMODE ̐ݒύX
//      Ȃ΂ȂȂB
//
//   N2Ԗڂ̃p[^ (option) w肵Ȃꍇ́Aȉ̒萔ɂ
//   ̏ڍׂ肷B
//
//   "FUNC_TYPE" ́Agp鏈֐̎ʂI邽߂ɒ`B
//   _TDSUDrvXxxx() gpꍇ́AHSMS-SS Active ڑ̏ꍇ̐ڑs
//   KvȂ̂ŁA{ł "SECS_MODE" ̐ݒKvȂB
//   TDS ́ASECS-1 ڑ or HSMS-SS ڑ̐ؑւ́Aݒt@C (.ini) ɂčs
//   ߁A֐Ƃ _TDSUDrvXxxx() gṕA[U`oł́A
//   ڑiSECS-1/HSMS-SS  HSMS-SS Passive/ActivejmKv
//   ȂB _TDSUDrvXxxx() gpꍇ́ACallback ͎͂ōs
//   KvB
//
//   ȉŒ` "MSSG_USE_FILE"A"MSSG_DISP_TYPE" ̒lύXƁA{`o
//   ɂĕ\ SECS bZ[W̏o͌`ύXłB
//   "MSSG_USE_NEXTL" ̒lύXƁAbZ[We͂ۂɁAڒl
//   f[^ƂĎ擾邩A\Xg`Ŏ擾邩ύXłB
//   l "USE_RECVTHREAD"  =1 ƂƁAMsXbhNA
//   ̃XbhŎMbZ[W̕\AQbZ[W̏o͂ōsB
//   MXbhgpꍇA͑ipbZ[Wɗꂪ鎖ɗ
//   邱ƁB
//   "USE_SEND_CB" ̒lύXƁAMʎ擾p Callback ̎gp̗LύX
//   łB
//   "USE_USER_TRACE" ̒lύXƁAUser Trace o͗p Callback ̎gp̗L
//   ύXłB
//   "HEADER_BO"  1 ƂƁAReceiveASend Ɏ擾 SECS bZ[WE
//   wb_̃oCgI[_[ SECS bZ[W{̃lbg[NEoCgI[_[
//   ƂĎ擾łB
//
// =============================================================================

#include	"TDS.h"


#define		PO		fprintf(stdout
#define		PE		fprintf(stderr

#define		SECS_MODE	1		// SECS/HSMS mode
						// 0    : SECS-1
						// 1    : HSMS
						// (Note) See comments above
						// () 㕔RgQ
#define		UDRV_MASK  	0x8383ffff	// Mask value for UDrvOpen()
						// 0          : Set =0x49
						// 0x8383ffff : All event

#define		FUNC_TYPE	1		// Type of function used
						// ʐMɎgp֐̎
						// 0    : _TDSCommXxxxx
						// 1    : _TDSUDrvXxxxx
#define		HEADER_BO	0		// Header byte order
						// 擾wb_ Byte Order
						// 0    : System  Order
						// 1    : Network Order
#define		MSSG_DISP_TYPE	0x20		// SECS Message display format
						// 0x00 : TDS Format
						// 0x20 : SML Format
#define		MSSG_USE_NEXTL	1		// Use MssgNextL() or not
						// 0    : Not use
						// 1    : Use

#define		MSSG_USE_FILE	0x00		// Message definition file
						// 0x00 : Not use
						// 0x80 : Use to display item
						//	  names
						//	  gpčږ\
#define		USE_RECVTHREAD	0		// Execute receiving process on
						// its own thread
						// MOXbhŎs
						// 0	: No
						// 1	: Yes
#define		USE_SEND_CB	0		// Use CB for sending result
						// MʎMp Callback gp
						// 0	: Not Use
						// 1	: Use
#define		USE_USER_TRACE	0		// Use User trace call back
						// function or not
						// UserTracepCB gp邩ۂ
						// 0    : Not use
						// 1    : Use

#define		SYNC_STATUS	1		// Synchronize connection status
						// ڑԂ̓
						// 0    : No
						// 1    : Yes

#define		SEND_WAIT	200		// Message sending interval(ms)
						// bZ[WMԊu      (mb)
#define		SEND_NO		10		// Number of times until message
						// sending off
						// bZ[WMx݂܂ł̉
#define		SEND_SLEEP	1000		// Wait time		   (ms)
						// xݎ              (mb)
#define		ERROR_SLEEP	2000		// Wait time in error case (ms)
						// G[xݎ  (mb)

#define		PARAMFILE	"Sample.ini"


static int	FuncType	=FUNC_TYPE;		// argv[2]&0x0001
static int	HeaderBO	=HEADER_BO;		// argv[2]&0x0002
static int	MssgDispType	=MSSG_DISP_TYPE;	// argv[2]&0x0004
static int	MssgUseNextL	=MSSG_USE_NEXTL;	// argv[2]&0x0008
static int	MssgUseFile	=MSSG_USE_FILE;		// argv[2]&0x0010
static int	UseRecvThrd	=USE_RECVTHREAD;	// argv[2]&0x0020
static int	UseSendCB	=USE_SEND_CB;		// argv[2]&0x0040
static int	UseUserTrace	=USE_USER_TRACE;	// argv[2]&0x0080
static int	SyncStatus	=SYNC_STATUS;		// argv[2]&0x8000
static int	NoListOut	=0;			// argv[2]&0x0100

static int	SendWait	=SEND_WAIT;		// argv[3]
static int	SendSleep	=SEND_SLEEP;		// argv[4]

static int	Finish=0;	// End instruction to thread
				// Xbhւ̏Iw


static void	Host();
static void	Equip();



// =============================================================================
// Common functions ============================================================

#include	"SubFunctions.h"



// =============================================================================
// Receive processing thread ---------------------------------------------------
// MXbh ------------------------------------------------------------

#if defined	WINDOWS
static void
#else
static void*
#endif
RecvProcThread(
void			*par)
{
  TDSCBData		cbd;
  char			msg[4096];
  int			msz,cmd=0;

  cbd.vec=1;		cbd.msg=msg;
  msz=sizeof(msg);

  for(;Finish==0;){
    // Header Byte Order (HBO) conversion is performed by CBRecvProc() in the
    // same way as when TDSCommOpen() instructs use of Callback for receiving,
    // so treat it in the same way. (If Callback is called from TDS, do not
    // convert HBO.)  Therefore, HBO conversion is not instructed here.
    // wb_EoCgI[_[ΐA_TDSCommOpen() ŎM Callback gp
    // wꍇƓliCallback  Call ́Awb_EoCgI[_[ϊ
    // ł͂ȂjCBRecvProc() ōŝŁAƓl̈ƂȂ悤A
    // ł̓wb_EoCgI[_[ϊwȂB
    //if(HeaderBO!=0)	cmd=0x2000;
    //else		cmd=0x0000;
    if(FuncType	 ==0){
      cbd.rtn=
	_TDSCommRecv(Fd,cmd,&cbd.devid,&cbd.sf,&cbd.xid,cbd.msg,msz,&cbd.thd);
    }else{
      cbd.rtn=
	_TDSUDrvRecv(Fd,cmd,&cbd.devid,&cbd.sf,&cbd.xid,cbd.msg,msz,&cbd.thd);
    }
    if(cbd.rtn	== -E_NODATA)	Sleep(100);
    else{
      if((-1000)<cbd.rtn && cbd.rtn<(-959))	cbd.req=(-cbd.rtn)-900;
      else					cbd.req=0;
      CBRecvProc(par,&cbd);
      if(cbd.rtn== -E_ACCES)	Break=1;
  } }
}



// =============================================================================
// Main process ----------------------------------------------------------------

int
main(
int			argc,
char			*argv[])
{
  int			opt=0;

  // ---------------------------------------------------------------------------
  { TDSVersion		inf;	// Display TDS information
    int			bits;	// This code has nothing to do with running TDS.
    bits=_TDSGetVersion(&inf);
    PO,"Version = %-20s %-20s\n"	,inf.version[0],inf.version[1]);
    PO,"Date    = %-20s %-20s\n"	,inf.verdate[0],inf.verdate[1]);
    PO,"Compile = %-20s %-20s\n"	,inf.compile[0],inf.compile[1]);
    PO,"#of Bit = %d\n"			,bits);
  }
  // ---------------------------------------------------------------------------

  // ---------------------------------------------------------------------------
  if(FuncType		!=0)		opt|=0x0001;
  if(HeaderBO		!=0)		opt|=0x0002;
  if(MssgDispType	!=0x00)		opt|=0x0004;
  if(MssgUseNextL	!=0)		opt|=0x0008;
  if(MssgUseFile	!=0x00)		opt|=0x0010;
  if(UseRecvThrd	!=0)		opt|=0x0020;
  if(UseSendCB		!=0)		opt|=0x0040;
  if(UseUserTrace	!=0)		opt|=0x0080;
  if(NoListOut		!=0)		opt|=0x0100;
  if(SyncStatus		!=0)		opt|=0x8000;
  if(argc>2){			// Analysis of running parameters
    if	   (argv[2][0]=='x' || argv[2][0]=='X')	 sscanf(&argv[2][1],"%x",&opt);
    else if(argv[2][0]=='0' &&
	   (argv[2][1]=='x' || argv[2][1]=='X')) sscanf(&argv[2][2],"%x",&opt);
    else					 sscanf(&argv[2][0],"%d",&opt);
    if(opt&0x0001) FuncType    =1;	else FuncType    =0;
    if(opt&0x0002) HeaderBO    =1;	else HeaderBO    =0;
    if(opt&0x0004) MssgDispType=0x20;	else MssgDispType=0x00;
    if(opt&0x0008) MssgUseNextL=1;	else MssgUseNextL=0;
    if(opt&0x0010) MssgUseFile =0x80;	else MssgUseFile =0x00;
    if(opt&0x0020) UseRecvThrd =1;	else UseRecvThrd =0;
    if(opt&0x0040) UseSendCB   =1;	else UseSendCB   =0;
    if(opt&0x0080) UseUserTrace=1;	else UseUserTrace=0;
    if(opt&0x0100) NoListOut   =1;	else NoListOut   =0;
    if(opt&0x8000) SyncStatus  =1;	else SyncStatus	 =0;
  }
  if(argc>3)	sscanf(argv[3],"%d"	,&SendWait);
  if(argc>4)	sscanf(argv[4],"%d"	,&SendSleep);

  PO,"option      =0x%02x,%d\n"		,opt,opt);
  PO,"FuncType    =%d\n"		,FuncType);
  PO,"HeaderBO    =%d\n"		,HeaderBO);
  PO,"MssgDispType=0x%02x\n"		,MssgDispType);
  PO,"MssgUseNextL=%d\n"		,MssgUseNextL);
  PO,"MssgUseFile =0x%02x\n"		,MssgUseFile);
  PO,"UseRecvThrd =%d\n"		,UseRecvThrd);
  PO,"UseSendCB   =%d\n"		,UseSendCB);
  PO,"UseUserTrace=%d\n"		,UseUserTrace);
  PO,"NoListOut   =%d\n"		,NoListOut);
  PO,"SyncStatus  =%d\n"		,SyncStatus);
  PO,"SendWait    =%d\n"		,SendWait);
  PO,"SendSleep   =%d\n"		,SendSleep);
  // ---------------------------------------------------------------------------

  if(argc<2 || (argv[1][0]!='h' && argv[1][0]!='e')){
    PE,"\nUsage: %s {host|equip} [option [send_wait [send_sleep]]]\n",argv[0]);
    PE,"\n  option\n");
    PE,  "   F       8 7654 3210\n");
    PE,  "  +----+----+----+----+\n");
    PE,  "   |       | |||| |||+-- Function to use\n");
    PE,  "   |       | |||| |||    =0: _TDSCommXxxxx()           1:_TDSUDrvXxxx()\n");
    PE,  "   |       | |||| ||+--- SECS Header Byte order\n");
    PE,  "   |       | |||| ||     =0: System order              1: Network order\n");
    PE,  "   |       | |||| |+---- Format for displaying messages in SECS list format\n");
    PE,  "   |       | |||| |      =0: TDS Format                1: SML Format\n");
    PE,  "   |       | |||| +----- Whether to use MssgNextL() for SECS message display\n");
    PE,  "   |       | ||||        =0: Not use                   1: Use\n");
    PE,  "   |       | |||+------- Whether to use the SECS message definition file\n");
    PE,  "   |       | |||         =0: Not use                   1: Use\n");
    PE,  "   |       | ||+-------- Whether to execute SECS message reception in a dedicated thread\n");
    PE,  "   |       | ||          =0: No                        1: Yes\n");
    PE,  "   |       | |+--------- Whether to use the Callback function for receiving transmission results\n");
    PE,  "   |       | |           =0: Not use                   1: Use\n");
    PE,  "   |       | +---------- Whether user-specified trace output function is used\n");
    PE,  "   |       |             =0: Not use                   1: Use\n");
    PE,  "   |       +------------ Suppress Message structure output for testing\n");
    PE,  "   |                     =0: No                        1: Yes\n");
    PE,  "   +-------------------- Synchronize connection status\n");
    PE,  "                         =0: No                        1: Yes\n");
    PE,  "  send_wait   : Message sending interval     (%5dms)\n",SendWait);
    PE,  "  send_sleep  : Wait time                    (%5dms)\n",SendSleep);
    PE,"\n");
    exit(1);
  }

  signal(SIGINT,CBSignal);
//InitializeCriticalSection(&Cs0);
  InitializeCriticalSection(&Cs1);

  if(argv[1][0]=='h')	Host ();
  else			Equip();

//DeleteCriticalSection(&Cs0);
  DeleteCriticalSection(&Cs1);
  exit(0);
}



// =============================================================================
// Host side process -----------------------------------------------------------

static void
Host()
{
  int			(*cbrecv )(void*,TDSCBData*)=0;
  int			(*cbsend )(void*,TDSCBData*)=0;
  int			(*cbtrace)(void*,TDSTRData*)=0;
  char			mname[64];
  int			req,rtn,omd,mno,stat,i;

  if(UseRecvThrd ==0)	cbrecv	= CBRecvProc;
  if(UseSendCB   !=0)	cbsend	= CBSend;
  if(UseUserTrace!=0)	cbtrace	= CBTrace;
  if(SyncStatus  ==0)	omd	= 0x0002;
  else			omd	= 0x1002;
  if(FuncType	 ==0){
    Fd=_TDSCommOpen(omd,PARAMFILE,"HOST",cbrecv,0,cbsend,0,cbtrace,0);
  }else{
    Fd=_TDSUDrvOpen(omd,PARAMFILE,"HOST",UDRV_MASK);
  }
  if(Fd								< 0) goto Exit;
  PO,"(H) Opened (%d)\n",Fd);	OType=0;
  if((MssgUseFile&0x80)!=0) Md=_TDSMDMssgInitialize(0x4000,Fd,0);

	// When _TDSUDrvXxxx() is used and when receiving process is specified
	// in the receiving thread, start the receiving process thread.
  	// _TDSUDrvXxxx() gp̏ꍇ y MpXbhł̎Mw̏ꍇ
	// ͎MXbhN
  if(FuncType	 !=0 || UseRecvThrd!=0){
    _beginthread(RecvProcThread,0,0);
  }

  if(SECS_MODE!=0){
	// In case of HSMS, wait for Select state.
	// For SECS-1 connection, the following connection status confirmation
	// processing may be performed.
	// (Note 1) If connection status is not synchronized with (omd&0x1000)==0
	// at CommOpen() or UDrvOpen(), connection status change report must be
	// received with CommRecv() or UDrvRecv().  (The processing is not
	// described here.)
	// If you are using a receive thread with UseRecvThrd!=0, you don't need,
	// because that thread will receive connection state change reports.
	// (Note 2) In the first place, even if it is not in Selected state, if
	// you think that "_TDSCommSend() etc. will only result in an error
	// (E_ILSEQ)", at least in this AP, this "waiting for Selected state"
	// operation is not necessary.
	//
	// HSMS ڑ̏ꍇASelect ԂɂȂ̂҂B
	// SECS-1 ڑ̏ꍇȉ̐ڑXe[^XmF͍sĂ܂ȂB
	// (1) CommOpen()AUDrvOpen()  (omd&0x1000)==0 ƂĐڑXe[^X
	// ̓sȂꍇ́ACommRecv()AUDrvRecv() ŁAڑԕω񍐂
	// MȂ΂ȂȂB
	// UseRecvThrd!=0 ŎMXbhgpĂꍇ́ÃXbhɂ
	// ڑԕω񍐂M̂ŁA̕Kv͂ȂB
	// (2) A Selected ԂɂȂĂȂĂAuȍ~
	// _TDSCommSend() G[ (E_ILSEQ) ɂȂ邾vƍl΁AȂƂ
	//  AP ł́ÁuSelected ԂɂȂ̂҂vƂ쎩̂Kv
	// ȂB
    for(;;){
	// CallbackIo always uses the receive CB, so there is no need to call
	// Recv() below.
	// CallbackIo ł́AɎM CB gp̂ŁAȉ Recv()  Call 
	// KvȂB
    //if(SyncStatus==0 && UseRecvThrd==0){
    //  if(FuncType==0)	_TDSCommRecv(Fd,0x0000,&did,&sf,&xid,msg,msz,&hd);
    //  else		_TDSUDrvRecv(Fd,0x0000,&did,&sf,&xid,msg,msz,&hd);
    //}
      if(FuncType==0)	rtn=_TDSCommStatus(Fd,0x0000);
      else		rtn=_TDSUDrvStatus(Fd,0x0000);
      if(rtn							< 0) goto Exit;
      if(rtn							==3) break;
      Sleep(100);
    }
    printf("(H) Selected\n");
  }

// (Note 3) In other words, when SyncStatus!=0, the following code is OK for
//    both SECS-1 and HSMS.
// (3)  SyncStatus!=0 ̏ꍇASECS-1AHSMS Ɉȉ̃R[hłnjB
//for(;;){
//  if(FuncType==0)	rtn=_TDSCommStatus(Fd,0x0000)
//  else		rtn=_TDSUDrvStatus(Fd,0x0000)
//  if(rtn							< 0) goto Exit;
//  if(rtn							==3) break;
//  Sleep(100);
//}

  for(;;){
    req=99;	rtn=0;
    PE,"Req (0:Exit 2:Send 3:Continuous Send 4:Statistic Report) : ");
    scanf("%d",&req);
    if      (req==0){						break;

    }else if(req==2){
      char			*frm0="1:S1F1 2:S1F21 3:S2F49";
      if((MssgUseFile&0x80)!=0)  frm0="1:S1F1 2:S1F21 3:S2F49   9:Any";
      PE,"Message(%s) : "	,frm0);			scanf("%d",&mno);
      switch(mno){
	case 1: rtn=SendS1F1 ();				break;
	case 2: rtn=SendS1F21();				break;
	case 3: rtn=SendS2F49();				break;
	case 9: if((MssgUseFile&0x80)!=0){
		  PE,"Message Name : ");		scanf("%s",mname);
		  rtn=SendNamedMssg(0,0,mname);
		}						break;
      }

    }else if(req==3){
      for(Break=i=0;Break==0;i++){
	Sleep(SendWait);
	if(FuncType==0)	stat=_TDSCommStatus(Fd,0);
	else		stat=_TDSUDrvStatus(Fd,0);
	if(stat==3){
	  if((rtn=SendS2F49())	< 0){
	    PE,"(H) Send error (%d)\n"	,rtn);	Sleep(ERROR_SLEEP);
	    if(rtn		==-E_ACCES)	Break=1;
	  }
	  if((i%SEND_NO)		==0)	Sleep(SendSleep);
      } }
      rtn=0;

    }else if(req==4){
      TDSSInfoSP	sp;
      int		noe,nort,i,s,f,d;
      for(noe=i=0;i<2;i++){
	if(i==0) PO,"\n[Sent Primary Message Table]\n");
	else	 PO,"\n[Received Primary Message Table]\n");
	PO, "   SC  FC: Primary: Second.:Async T3TO"
	    ": S9F1   F3   F5   F7   F9  F11  F13:  Total.ms (   Ave. ms)\n");
	for(s=0;s<128;s++){	for(f=0;f<256;f++){
	  if(FuncType	==0)	rtn=_TDSCommSInfoGet(Fd,i,(s<<8)|f,&sp);
	  else			rtn=_TDSUDrvSInfoGet(Fd,i,(s<<8)|f,&sp);
	  if(rtn	==0){	noe++;
	    if((d=f)	!=0)	f++;
	    if((nort=sp.nort)==0) nort=1;
	    PO,	"  %3d %3d:%8d:%8d:%5d%5d"
		":%5d%5d%5d%5d%5d%5d%5d:%8lldms (%8.1fms)\n"
		  ,s,d  ,sp.nox  ,sp.nort ,sp.norf,sp.not3
			,sp.nof1 ,sp.nof3 ,sp.nof5,sp.nof7,sp.nof9
			,sp.nof11,sp.nof13,sp.trtok,(float)sp.trtok/(float)nort);
	  }else{
	    if(f	!=0)	f++;
      } }}}
      if(noe==0){
	PO,"\nSet STATISINFO=1 in the .ini file when using this function.\n\n");
      }else	PO,"\n");
      rtn=0;
    }

    if(rtn<(-999) || ((-900)<rtn && rtn<0) ){
      PE,"(H) Send error (%d)\n"	,rtn);
  } }

Exit:
  Finish=1;
  if(Fd> 0){
    if(FuncType	==0)	_TDSCommClose(Fd,0);
    else		_TDSUDrvClose(Fd,0);
  }else			PE,"(H) Error (%d)\n",Fd);
  if(Md> 0)		_TDSMDMssgTerminate(Md,0);
}



// =============================================================================
// Equipment side process ------------------------------------------------------

static void
Equip()
{
  int			(*cbrecv )(void*,TDSCBData*)=0;
  int			(*cbsend) (void*,TDSCBData*)=0;
  int			(*cbtrace)(void*,TDSTRData*)=0;
  char			mname[64];
  int			req,rtn,omd,mno,stat,i;

  if(UseRecvThrd ==0)	cbrecv	= CBRecvProc;
  if(UseSendCB   !=0)	cbsend	= CBSend;
  if(UseUserTrace!=0)	cbtrace	= CBTrace;
  if(SyncStatus  ==0)	omd	= 0x0002;
  else			omd	= 0x1002;
  if(FuncType	 ==0){
    Fd=_TDSCommOpen(omd,PARAMFILE,"EQUIP",cbrecv,0,cbsend,0,cbtrace,0);
  }else{
    Fd=_TDSUDrvOpen(omd,PARAMFILE,"EQUIP",UDRV_MASK);
  }
  if(Fd								< 0) goto Exit;

	// When _TDSUDrvXxxx() is used and when receiving process is specified
	// in the receiving thread, start the receiving process thread.
  	// _TDSUDrvXxxx() gp̏ꍇ y MpXbhł̎Mw̏ꍇ
	// ͎MXbhN
  if(FuncType	 !=0 || UseRecvThrd!=0){
    _beginthread(RecvProcThread,0,0);
  }

  PO,"(E) Opened (%d)\n",Fd);	OType=1;
  if((MssgUseFile&0x80)!=0) Md=_TDSMDMssgInitialize(0x4000,Fd,0);

  if(SECS_MODE !=0){	// In case of HSMS
    if(FuncType==0){	// In case of use TDSCommXxxxx()
      if(_TDSCommSend(Fd,0x0100,0,0,0,0,0,0)			< 0) goto Exit;
      printf("(E) Connected\n");
      if(_TDSCommSend(Fd,0x0200,0,0,0,0,0,0)			< 0) goto Exit;
      printf("(E) Select request\n");
    }

    for(;;){	// Wait for Select state. See the description for Host().
    		// Select ԂɂȂ̂҂B Host() ł̐QƂ邱ƁB
    //if(SyncStatus==0 && UseRecvThrd==0){
    //  if(FuncType==0)	_TDSCommRecv(Fd,0x0000,&did,&sf,&xid,msg,msz,&hd);
    //  else		_TDSUDrvRecv(Fd,0x0000,&did,&sf,&xid,msg,msz,&hd);
    //}
      if(FuncType==0)	rtn=_TDSCommStatus(Fd,0x0000);
      else		rtn=_TDSUDrvStatus(Fd,0x0000);
      if(rtn							< 0) goto Exit;
      if(rtn							==3) break;
      Sleep(100);
    }
    printf("(E) Selected\n");
  }

  for(;;){
    req=99;	rtn=0;
    PE,"Req (0:Exit 2:Send 3:Continuous Send 4:Statistic Report) : ");
    scanf("%d",&req);
    if      (req==0){						break;

    }else if(req==2){
      char			*frm0="1:S1F1 2:S1F21 3:S6F11";
      if((MssgUseFile&0x80)!=0)  frm0="1:S1F1 2:S1F21 3:S6F11   9:Any";
      PE,"Message(%s) : "	,frm0);			scanf("%d",&mno);
      switch(mno){
	case 1: rtn=SendS1F1 ();				break;
	case 2: rtn=SendS1F21();				break;
	case 3: rtn=SendS6F11();				break;
	case 9: if((MssgUseFile&0x80)!=0){
		  PE,"Message Name : ");		scanf("%s",mname);
		  rtn=SendNamedMssg(0,0,mname);
		}						break;
      }

    }else if(req==3){
      for(Break=i=0;Break==0;i++){
	Sleep(SendWait);
	if(FuncType==0)	stat=_TDSCommStatus(Fd,0);
	else		stat=_TDSUDrvStatus(Fd,0);
	if(stat==3){
	  if((rtn=SendS6F11())	< 0){
	    PE,"(E) Send error (%d)\n"	,rtn);	Sleep(ERROR_SLEEP);
	    if(rtn		==-E_ACCES)	Break=1;
	  }
	  if((i%SEND_NO)		==0)	Sleep(SendSleep);
      } }
      rtn=0;

    }else if(req==4){
      TDSSInfoSP	sp;
      int		noe,nort,i,s,f,d;
      for(noe=i=0;i<2;i++){
	if(i==0) PO,"\n[Sent Primary Message Table]\n");
	else	 PO,"\n[Received Primary Message Table]\n");
	PO, "   SC  FC: Primary: Second.:Async T3TO"
	    ": S9F1   F3   F5   F7   F9  F11  F13:  Total.ms (   Ave. ms)\n");
	for(s=0;s<128;s++){	for(f=0;f<256;f++){
	  if(FuncType	==0)	rtn=_TDSCommSInfoGet(Fd,i,(s<<8)|f,&sp);
	  else			rtn=_TDSUDrvSInfoGet(Fd,i,(s<<8)|f,&sp);
	  if(rtn	==0){	noe++;
	    if((d=f)	!=0)	f++;
	    if((nort=sp.nort)==0) nort=1;
	    PO,	"  %3d %3d:%8d:%8d:%5d%5d"
		":%5d%5d%5d%5d%5d%5d%5d:%8lldms (%8.1fms)\n"
		  ,s,d  ,sp.nox  ,sp.nort ,sp.norf,sp.not3
			,sp.nof1 ,sp.nof3 ,sp.nof5,sp.nof7,sp.nof9
			,sp.nof11,sp.nof13,sp.trtok,(float)sp.trtok/(float)nort);
	  }else{
	    if(f	!=0)	f++;
      } }}}
      if(noe==0){
	PO,"\nSet STATISINFO=1 in the .ini file when using this function.\n\n");
      }else	PO,"\n");
      rtn=0;
    }

    if(rtn<(-999) || ((-900)<rtn && rtn<0)){
      PE,"(E) Send error (%d)\n"	,rtn);
  } }

  // If you want to terminate (call _TDSCommClose() or _TDSUDrvClose()) like
  // this AP, you don't need the following "Separate request" processing.
  //  AP ̗lɁÂ܂܏I (_TDSCommClose()  _TDSUDrvClose()
  //  Call ) ̂ł΁Aȉ "Separate v" ͕KvȂB
  if(SECS_MODE	!=0){			// In case of HSMS, Shutdown process
					// HSMS ̏ꍇAؒf
    if(FuncType	==0)	rtn=_TDSCommStatus(Fd,0);
    else		rtn=_TDSUDrvStatus(Fd,0);
    if(rtn	==3){
    // Deselect request is not performed. (Of course you may go. However SEMI
    // claims that HSMS-SS does not perform Deselect request.)
    // Deselect request ͍sȂB (sĂ悢BSEMI ł HSMS-SS
    // ɂāADeselect request ͍sȂAƂĂB)
    //if(FuncType==0)	rtn=_TDSCommSend(Fd,0x0800,0,0,0,0,0,0);
    //else		rtn=_TDSUDrvSend(Fd,0x0800,0,0,0,0,0,0);
    //if(rtn							< 0) goto Exit;
    //PO,"(E) Deselected\n");
      if(FuncType==0)	rtn=_TDSCommSend(Fd,0x0900,0,0,0,0,0,0);
      else		rtn=_TDSUDrvSend(Fd,0x0900,0,0,0,0,0,0);
      if(rtn							< 0) goto Exit;
      PO,"(E) Separated\n");
  } }

Exit:
  Finish=1;
  if(Fd> 0){
    if(FuncType	==0)	_TDSCommClose(Fd,0);
    else		_TDSUDrvClose(Fd,0);
  }else			PE,"(E) Error (%d)\n",Fd);
  if(Md> 0)		_TDSMDMssgTerminate(Md,0);
}
