
//
// Trust Design LLC : SECS/HSMS Communication library
//
// (c) Copyright Trust Design LLC.  2010-2024.  All rights reserved.
//


// =============================================================================
//
// JVIo : Test and sample program
//
//   Construct a message in AP and simply send and receive SECS messages.
//
//   (Note) In this sample, when making an HSMS connection, it is assumed that
//	the HOST side executes Passive connection and the EQUIP side executes
//	Active connection.
//
//
// Starting method
//
//   java JVIo {h|e} [option [send_wait [send_sleep]]]
//   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//   h    : Refer to [HOST]  section of Sample.ini to determine operation
//   e    : Refer to [EQUIP] section of Sample.ini to determine operation
//
//   option : Options for setting the details of operation (optional)
//	      Specify as a decimal number or as a hexadecimal number with 0x or
//	      x as a prefix.
//	      The details of the specified value for each bit are shown below.
//    F		  54 3210
//   +----+----+----+----+
//    |		  || |||+-- Function to use
//    |		  || |||    =0: _TDSCommXxxxx()		1:_TDSUDrvXxxx()
//    |		  || ||+--- SECS Header Byte order
//    |		  || ||	    =0: System order		1: Network order
//    |		  || |+---- Format for displaying messages in SECS list format
//    |		  || |	    =0: TDS Format		1: SML Format
//    |		  || +----- Whether to use MssgNextL() for SECS message display
//    |		  ||	    =0: Not use			1: Use
//    |		  |+------- Whether to use the SECS message definition file
//    |		  |	    =0: Not use			1: Use
//    |		  +-------- Whether to execute SECS message reception in a
//    |			    dedicated thread
//    |			    =0: No			1: Yes
//    +-------------------- Synchronize the connection status with communication
//			    control processing thread
//			    =0: No			1: Yes
//
//   send_wait   : Transmission interval when sending continuous messages (ms).
//   send_sleep  : Time to rest for a short time after sending a given number
//                 of times (SEND_NO) when sending continuous messages (ms).
//
//   Normally, "JVIo h" and "JVIo e" both operate on same machine or
//   different machines to communicate with each other. The communication
//   partner may be BasicIo.exe or Callbackio.exe.
//
//   After startup, refer to the menu display, enter request code etc. and the
//   function is executed.
// 
//   When host starts up, it waits for a connection from equipment.
//   When device side is started, connection request and Select request are
//   automatically issued to host side.
//   On host side, repeat '1:Recv' processing, and confirm that display is
//   'STATUS=951: No data'. After that, exchange any message. On receiving side,
//   do not forget to reap received data until 'STATUS=951'.
//
//   The transaction ID used when sending secondary message uses that of primary
//   message received immediately before.
//
//   This sample omits some of abnormal processing.
//
//   "SECS_MODE" defined below is defined to determine whether or not connection
//   processing to Passive side in case of Active connection is performed when
//   using _TDJVSCommXxxx() as processing function.
//   (Note) Changing this value alone does not make TDS aware of the change in
//      SECS1 or HSMS.  TDS determines SECS1 or HSMS based on the setting of
//      SECSMODE in the configuration file (.ini).  To change this value, you
//      must change the setting of SECSMODE in .ini.
//
//   If the second parameter (option) at startup is not specified, the details
//   of operation are determined by the following constants.
//
//   "FUNC_TYPE" is defined to select type of processing function to be used.
//   If you use _TDJVSUDrvXxxx(), there is no need to perform connection
//   processing for HSMS-SS Active connection, so there is no need to set
//   "SECS_MODE" in principle.
//   Since TDS switches SECS-1 connection or HSMS-SS connection in setting file
//   (.ini), user AP does not need to know connection method (SECS-1or HSMS and
//   Passive/Active at HSMS) as long as _TDJVSUDrvXxxx() is used as processing
//   function.
//
//   By changing values of "MSSG_USE_FILE" and "MSSG_DISP_TYPE" defined below,
//   you can change output format of SECS message displayed by this AP.
//   By changing value of "MSSG_USE_NEXTL", it is possible to change whether
//   to acquire field values as data or display list format when analyzing
//   message contents.
//   Similarly, when "USE_RECVTHREAD" is set to =1, a thread for performing
//   receiving process is activated, and display of received message and output
//   of the secondary message are automatically performed in the thread.
//   When using the receiving thread, keep in mind that prompt message may be
//   distorted.
//   If "HEADER_BO" is set to 1, Byte Order of SECS message header obtained at
//   the time of Receive and Send can be obtained as the original network Byte
//   Order of the SECS message.
//
//
// =============================================================================
//
// JVIo : eXg y TvEvO
//
//   `oŃbZ[W\zAP SECS bZ[W̑MsB
//
//   () {Tvł HSMS ڑꍇAHOST  Passive ڑAEQUIP 
//	Active ڑsƉ肵ĂB
//
//
// N@
//
//   java JVIo {h|e} [option [send_wait [send_sleep]]]
//   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//   h    : Sample.ini  [HOST]  ZNVQƂ肷
//   e    : Sample.ini  [EQUIP]      :              :
//
//   option : ̏ڍׂݒ肷IvV (ȗ\)
//	      10i  ړƂ 0x  x t^ 16i
//	      Ŏw肷B
//	      ȉɎwl̃rbg̏ڍׂȉɎB
//    F		  54 3210
//   +----+----+----+----+
//    |		  || |||+-- gp֐
//    |		  || |||    =0: _TDSCommXxxxx()		1:_TDSUDrvXxxx()
//    |		  || ||+--- SECS Header Byte order
//    |		  || ||	    =0: System order		1: Network order
//    |		  || |+---- SECS Xg`ŃbZ[W\ꍇ̌`
//    |		  || |	    =0: TDS `		1: SML `
//    |		  || +----- SECS bZ[W\ MssgNextL() gp邩ۂ
//    |		  ||	    =0: gpȂ		1: gp
//    |		  |+------- SECS bZ[W`t@Cgp̗L
//    |		  |	    =0: gpȂ		1: gp
//    |		  +-------- SECS bZ[WMpXbhŎs邩ۂ
//    |			    =0: No			1: Yes
//    +-------------------- ڑԂʐM䕔ɓ
//			    =0: No			1: Yes
//
//   send_wait   : AbZ[WM̑MԊu (ms)
//   send_sleep  : AbZ[WMɏ̉ (SEND_NO) Mɏ
//                 Mx݂鎞 (ms)
//
//   ʏAJVIo h y JVIo e ̗A}VA͈قȂ}V
//   삳āAݒʐMsBʐḾABasicIo.exeACallbackio.exe ł
//   悢B
//
//   NAj\QƂAvR[h͂A@\sB
// 
//   zXg͋NƁAu̐ڑ҂BuNƁAI
//   ɃzXgɐڑvASelect v𔭍sBzXgł́A1:Recv 
//   JԂsA\uSTATUS=951 : No datavƂȂ̂mFB̌A
//   Cӂ̃bZ[ŴƂsBMł́AMf[^̊u
//   STATUS=951vɂȂ܂ōsƂYʂƁB
//
//   QbZ[W𑗐Mۂ̃gUNVhćAOɎMP
//   bZ[Ŵ̂gpB
//
//   {Tv́Aُ펞ꕔȗĂB
//
//   ȉŒ` "SECS_MODE" ́A֐Ƃ _TDJVSCommXxxx() gp
//   ꍇ Active ڑ̏ꍇ Passive ւ̐ڑ̎s̗L𔻒f
//   ߂ɒ`B
//   () ̒l̕ύX TDS  SECS1 or HSMS ̕ύXF킯ł͂ȂB 
//      TDS  SECS1 or HSMS 𔻒f̂͐ݒt@C (.ini)  SECSMODE 
//      ݒɂ̂ŁA̒lύXꍇ .ini  SECSMODE ̐ݒύX
//      Ȃ΂ȂȂB
//
//   N2Ԗڂ̃p[^ (option) w肵Ȃꍇ́Aȉ̒萔ɂ
//   ̏ڍׂ肷B
//
//   "FUNC_TYPE" ́Agp鏈֐̎ʂI邽߂ɒ`B
//   _TDJVSUDrvXxxx() gpꍇ́AHSMS-SS Active ڑ̏ꍇ̐ڑ
//   sKvȂ̂ŁA{ł "SECS_MODE" ̐ݒKvȂB
//   TDS ́ASECS-1 ڑ or HSMS-SS ڑ̐ؑւ́Aݒt@C (.ini) ɂčs
//   ߁A֐Ƃ _TDJVSUDrvXxxx() gṕA[U`oł́A
//   ڑiSECS-1/HSMS-SS  HSMS-SS Passive/ActivejmKv
//   ȂB
//
//   ȉŒ` "MSSG_USE_FILE"A"MSSG_DISP_TYPE" ̒lύXƁA{`o
//   ɂĕ\ SECS bZ[W̏o͌`ύXłB
//   "MSSG_USE_NEXTL" ̒lύXƁAbZ[We͂ۂɁAڒl
//   f[^ƂĎ擾邩A\Xg`Ŏ擾邩ύXłB
//   l "USE_RECVTHREAD"  =1 ƂƁAMsXbhNA
//   ̃XbhŎMbZ[W̕\AQbZ[W̏o͂ōsB
//   MXbhgpꍇA͑ipbZ[Wɗꂪ鎖ɗ
//   邱ƁB
//   "HEADER_BO"  1 ƂƁAReceiveASend Ɏ擾 SECS bZ[WE
//   wb_̃oCgI[_[ SECS bZ[W{̃lbg[NEoCgI[_[
//   ƂĎ擾łB
//
// =============================================================================

import		java.lang.*;
import		java.util.*;
import		java.io.*;
import		java.nio.*;

import static	TDL.TDS.TDJVS.*;


public class	JVIo{


// -----------------------------------------------------------------------------

static final int	E_BADF		=9;
static final int	E_BUSY		=16;
static final int	E_NOMEM		=12;
static final int	E_NODEV		=19;
static final int	E_2BIG		=7;

static final int	E_NOTCONNECT	=999;
static final int	E_DESELECT	=998;
static final int	E_REJECT	=997;
static final int	E_SEPARATE	=996;
static final int	E_SELECT	=992;
static final int	E_CONNECT	=991;
static final int	E_RETRYOVER	=989;
static final int	E_T8TIMEDOUT	=988;
static final int	E_T7TIMEDOUT	=987;
static final int	E_T6TIMEDOUT	=986;
static final int	E_T5TIMEDOUT	=985;
static final int	E_T4TIMEDOUT	=984;
static final int	E_T3TIMEDOUT	=983;
static final int	E_T2TIMEDOUT	=982;
static final int	E_T1TIMEDOUT	=981;
static final int	E_ILLBLOCK	=980;
static final int	E_NODATA	=951;

static final int	XMSGSIZE	=(1024*1024);


// -----------------------------------------------------------------------------

static final int	SECS_MODE	=1;	// SECS/HSMS mode
						// 0    : SECS-1
						// 1    : HSMS
						// (Note) See comments above
						// () 㕔RgQ
static final int	UDRV_MASK  =0x8383ffff;	// Mask value of UDrvOpen()
						// 0          : Set =0x49
						// 0x8383ffff : All event

static final int	FUNC_TYPE	=1;	// Type of function used
						// ʐMɎgp֐̎
						// 0    : _TDJVSCommXxxxx
						// 1    : _TDJVSUDrvXxxxx
static final int	HEADER_BO	=0;	// Header byte order
						// 擾wb_ Byte Order
						// 0    : System  Order
						// 1    : Network Order
static final int	MSSG_DISP_TYPE	=0x20;	// SECS Message display format
						// 0x00 : TDS Format
						// 0x20 : SML Format
static final int	MSSG_USE_NEXTL	=1;	// Use MssgNextL() or not
						// 0    : Not use
						// 1    : Use

static final int	MSSG_USE_FILE	=0x00;	// Message definition file
						// 0x00 : Not use
						// 0x80 : Use to display item
						//	  names
						//	  gpčږ\
static final int	USE_RECVTHREAD	=1;	// Execute receiving process on
						// its own thread
						// MOXbhŎs
						// 0    : No
						// 1    : Yes

static final int	SYNC_STATUS	=1;	// Synchronize connection status
						// ڑԂ̓
						// 0    : No
						// 1    : Yes

static final int	SEND_WAIT	=200;	// Message sending interval (ms)
static final int	SEND_NO		=10;	// #of times until message
						// transmission break
static final int	SEND_SLEEP	=1000;	// Waiting time		    (ms)
static final int	ERROR_SLEEP	=1000;	// Waiting time when an  error
						// occurs		    (ms)

static final String	PARAMFILE	="Sample.ini";


// -----------------------------------------------------------------------------

static int		FuncType	=FUNC_TYPE;		// argv[1]&0x0001
static int		HeaderBO	=HEADER_BO;		// argv[1]&0x0002
static int		MssgDispType	=MSSG_DISP_TYPE;	// argv[1]&0x0004
static int		MssgUseNextL	=MSSG_USE_NEXTL;	// argv[1]&0x0008
static int		MssgUseFile	=MSSG_USE_FILE;		// argv[1]&0x0010
static int		UseRecvThrd	=USE_RECVTHREAD;	// argv[1]&0x0020
static int		SyncStatus	=SYNC_STATUS;		// argv[1]&0x8000

static int		SendWait	=SEND_WAIT;		// argv[2]
static int		SendSleep	=SEND_SLEEP;		// argv[3]

static int		Fd    =0;	// Communication identifier
static int		Md    =0;	// Message analysis identifier
static int		OType =0;	// Operation type  (0:Host  1:Equipment)
static int		Break =0;	// End instruction to thread
					// Xbhւ̏Iw

static int		cnt249=0;
static int		cnt250=0;
static int		cnt611=0;


// =============================================================================
// Common functions ============================================================

static private void
printf(
String		form,
Object...	arg)
{
  try{
    System.out.printf(form,arg);
  }catch(UnknownFormatConversionException	err){
    System.out.printf("Format error\n");
  }catch(Exception				err){
    System.out.printf("Printf error\n");
  }
}


// -----------------------------------------------------------------------------

static private void
Sleep(
int		msec)
{
  try{
    Thread.sleep((long)msec);
  }catch(InterruptedException	err){
  }catch(Exception		err){
  }finally{
  }
}


// -----------------------------------------------------------------------------

static private String
ReadLine(			//  o : Input string
BufferedReader	in,		// i  : Input descriptor
String		msg)		// i  : Input promotion message
{
  String	line;

  for(;;){
    printf("%s"	,msg);
    try{			line=in.readLine();
    }catch(Exception		err){			continue;
    }							break;
  }

  return(line);
}


static private int
ReadInt(			//  o : Input value
BufferedReader	in,		// i  : Input descriptor
String		msg,		// i  : Input promotion message
int		base)		// i  : Radix of integer
{
  String	line;
  int		val;

  for(;;){
    line=ReadLine(in,msg);
    try{			val=Integer.parseInt(line,base);
    }catch(Exception		err){			continue;
    }							break;
  }

  return(val);
}


// -----------------------------------------------------------------------------
// Display SECS messages on standard output ------------------------------------
// SECS bZ[WWo͂ɕ\B ---------------------------------------

static private void
DispSECSMssg(			//
int		tp,		// i  : Message type
				//	=0 : Transmission result
				//	 1 : Received message
				//	 2 : Send message
byte		hd[],		// i  : SECS Message Header
int		did,		// i  : SECS Message Device ID
int		sf,		// i  : SECS Message SF-Code
int		xid,		// i  : SECS Message Transaction ID
byte		msg[],		// i  : SECS Message strage area
				//	     (Header not included)
int		len)		// i  : SECS Message byte length
{
  String	ctp[]={"SRES","RECV","SEND"};
  String	rbit,wbit,sdid,sfcode,c;
  StringBuffer	mname=new StringBuffer(),sitem=new StringBuffer();
  ByteBuffer	pm;
  String	vs0[]={""},iname;
  byte		vu1[]=new byte  [256];
  short		vi2[]=new short [128];
  int		vi4[]=new int   [ 64];
  long		vi8[]=new long  [ 32];
  float		vf4[]=new float [ 64];
  double	vf8[]=new double[ 32];
  byte		msg1[];
  long		ul;
  int		form[]=new int[1],sz[]=new int[1],noi[]=new int[1];
  int		rtn,md=0,dp=MssgUseFile|MssgDispType,fm,la;
  int		no0,no1,no2,i,j;

  rbit=" ";	if((did&0x8000)!=0)	rbit="R";
  wbit=" ";	if((sf &0x8000)!=0)	wbit="W";
  if(did< 0)	sdid=String.format("%6d"	,did);
  else		sdid=String.format("0x%04x"	,did);
  sfcode=String.format("S%dF%d",(sf&0x7f00)/0x0100,sf&0xff);
  printf("[%s]  Dev=%-6s  %-8s  %s%s  XId=0x%04x.%04x  Len=%4d"+
	 "  Head=0x%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x\n"
		,ctp[tp],sdid,sfcode,rbit,wbit
		,(xid>>16)&0xffff,xid&0xffff,len
		,hd[0],hd[1],hd[2],hd[3],hd[4],hd[5],hd[6],hd[7],hd[8],hd[9]);

  if((MssgUseFile&0x80)!=0)	fm =0x8000;	// Use message definition file
  else				fm =0x0000;	// Not use

  if(len>0){
    if(tp==1){	dp|=0x3000;	fm|=0x3000;}	// In case of receiving
    else{	dp|=0x2000;	fm|=0x2000;}	// In case of sending
    pm=ByteBuffer.allocate(XMSGSIZE);	pm.order(ByteOrder.nativeOrder());
    msg1=pm.array();

    if((md=_TDJVSMssgFind(fm,msg,len,Fd,hd,mname))	> 0){
      if(mname.length()	> 0){
	printf("[%s]\n",mname.toString());	// Message name
      }

      for(la=0;;){
	if(MssgUseNextL==0){			// Get item value
	  rtn=_TDJVSMssgNext(md,0,msg,form,sz,noi,pm,XMSGSIZE);
	  form[0]&=077;
	  if(rtn						< 0)	break;
	  printf("%03o:%d*%3d:"	,form[0],sz[0],noi[0]);
	  switch(form[0]){
	    case 000: printf("L [%2d]\n",noi[0]);		break;
	    case 010: printf("B [%2d]"	,noi[0]);
		      for(c="=",i=0;i<noi[0];i++){
		        printf("%s0x%02x"
				,c,((int)msg1[i]&0xff));	if(i==0) c=",";
		      } printf("\n");				break; 
	    case 011: printf("T [%2d]"	,noi[0]);
		      for(c="=",i=0;i<noi[0];i++){
		        printf("%s%d"	,c,msg1[i]);		if(i==0) c=",";
		      } printf("\n");				break; 
	    case 020: c=new String(msg1,0 ,noi[0]);
		      printf("A [%2d]=%s.\n",noi[0],c);	c=null;	break;
	    case 021: c=new String(msg1,0 ,noi[0]);
		      printf("J [%2d]=%s.\n",noi[0],c);	c=null;	break;
	    case 022: c=new String(msg1,0 ,noi[0]);
		      printf("K [%2d]=%s.\n",noi[0],c);	c=null;	break;
	    case 030: printf("I8[%2d]"	,noi[0]);
		      for(c="=",i=0;i<noi[0];i++){
		        printf("%s%d"	,c,pm.getLong	(i*8));	if(i==0) c=",";
		      } printf("\n");				break;
	    case 031: printf("I1[%2d]"	,noi[0]);
		      for(c="=",i=0;i<noi[0];i++){
		        printf("%s%d"	,c,msg1[i]);		if(i==0) c=",";
		      } printf("\n");				break; 
	    case 032: printf("I2[%2d]"	,noi[0]);
		      for(c="=",i=0;i<noi[0];i++){
		        printf("%s%d"
				,c,(int)pm.getShort	(i*2));	if(i==0) c=",";
		      } printf("\n");				break; 
	    case 034: printf("I4[%2d]"	,noi[0]);
		      for(c="=",i=0;i<noi[0];i++){
		        printf("%s%d"	,c,pm.getInt	(i*4));	if(i==0) c=",";
		      } printf("\n");				break; 
	    case 040: printf("F8[%2d]"	,noi[0]);
		      for(c="=",i=0;i<noi[0];i++){
		        printf("%s%22.17e"
				,c,pm.getDouble		(i*8));	if(i==0) c=",";
		      } printf("\n");				break; 
	    case 044: printf("F4[%2d]"	,noi[0]);
		      for(c="=",i=0;i<noi[0];i++){
		        printf("%s%13.8e",c,pm.getFloat	(i*4));	if(i==0) c=",";
		      } printf("\n");				break; 
	    case 050: printf("U8[%2d]"	,noi[0]);
		      for(c="=",i=0;i<noi[0];i++){
		        printf("%s%d"	,c,pm.getLong	(i*8));	if(i==0) c=",";
		      } printf("\n");				break;
	    case 051: printf("U1[%2d]"	,noi[0]);
		      for(c="=",i=0;i<noi[0];i++){
			ul=((long)msg1[i])&0xffL;
		        printf("%s%d"	,c,ul);			if(i==0) c=",";
		      } printf("\n");				break; 
	    case 052: printf("U2[%2d]"	,noi[0]);
		      for(c="=",i=0;i<noi[0];i++){
			ul=((long)pm.getShort(i*2))&0xffffL;
		        printf("%s%d"	,c,ul);			if(i==0) c=",";
		      } printf("\n");				break; 
	    case 054: printf("U4[%2d]"	,noi[0]);
		      for(c="=",i=0;i<noi[0];i++){
			ul=((long)pm.getInt  (i*4))&0xffffffffL;
		        printf("%s%d"	,c,ul);			if(i==0) c=",";
		      } printf("\n");				break; 
	    default:  printf("\n");				break;
	  }

	}else{					// Get in list format
						// Xg\`Ŏ擾
	  rtn=_TDJVSMssgNextL(md,dp,msg,form,noi,mname);
	  form[0]&=077;
	  if(rtn						< 0)	break;
	  if((dp&0x70)>0x10){			// In case of SML
	    for(  i=la;i>rtn;i--){
	      for(j= 0;j<i  ;j++) printf("  ");
	      printf(">\n");			// Display '>'
	  } }
	  la=rtn;				// Save current hierarchy
						// ݂̊Kwۑ
	  printf("  %s\n",mname.toString());	// Display acquired field value
						// 擾ڒl\
	  if((dp&0x70)>0x10 && form[0]==000 && noi[0]==0){
	    for(  j= 0;j<=la;j++) printf("  ");	// '>' Processing for L0
	    printf(">\n");			// L0 ̏ꍇ '>' 
      } } }
      if(MssgUseNextL!=0){
	if((dp&0x70)>0x10){
	  for(  i=la;i>0;i--){
	    for(j= 0;j<i;j++)	  printf("  ");	// Show remaining '>'
	    printf(">\n");			// c '>' \
      } } }
      _TDJVSMssgExit(md,0,msg);
    }

    if((MssgUseFile&0x80)!=0  &&  MssgUseNextL	==0){
		// Refer to the comment in corresponding section of SubFuncsion.h
		// SubFuncsion.h ̊Yӏ̃RgQƂ邱
      sf&=0x7fff;
      if((rtn=_TDJVSMDMssgFind(Md,0,msg,len,sf,mname))		 > 0){
	printf("Message Definition Name = '%s'\n"	,mname);

	if	(sf	==0x0115){		// S1F21
	  if((no0=_TDJVSMDMssgNext(Md,0,msg,"VB"       ,256,vu1))>=0){
	    printf("VB          [%2d]=0x"	,no0);
	    for(i=0;i<no0;i++) printf("%02x "	,vu1[0]);	printf("\n");
	  }
	  if((no0=_TDJVSMDMssgNext(Md,0,msg,"MDLN"	,80,vs0))>=0){
	    printf("MDLN        [%2d]='%s'\n"	,no0	,vs0[0]);
	  }
	  if((no0=_TDJVSMDMssgNext(Md,0,msg,"SOFTREV"	,80,vs0))>=0){
	    printf("SOFTREV     [%2d]='%s'\n"	,no0	,vs0[0]);
	  }
	  if((no0=_TDJVSMDMssgNext(Md,0,msg,"VI4"	,64,vi4))>=0){
	    printf("VI4         [%2d]="		,no0);
	    for(i=0;i<no0;i++) printf("%d "	,vi4[i]);	printf("\n");
	  }
	  if((no0=_TDJVSMDMssgNext(Md,0,msg,"VF4"	,64,vf4))>=0){
	    printf("VF4         [%2d]="		,no0);
	    for(i=0;i<no0;i++) printf("%e "	,vf4[i]);	printf("\n");
	  }

	}else if(sf	==0x0116){		// S1F22
	  if((no0=_TDJVSMDMssgNext(Md,0,msg,"VT"       ,256,vu1))>=0){
	    printf("VT          [%2d]=0x"	,no0);
	    for(i=0;i<no0;i++) printf("%02x "	,vu1[i]);	printf("\n");
	  }
	  if((no0=_TDJVSMDMssgNext(Md,0,msg,"COMMENT"	,80,vs0))>=0){
	    printf("COMMENT     [%2d]='%s'\n"	,no0	,vs0[0]);
	  }
	  if((no0=_TDJVSMDMssgNext(Md,0,msg,"AU8"	,32,vi8))>=0){
	    printf("AU8         [%2d]="		,no0);
	    for(i=0;i<no0;i++) printf("%d "	,vi8[i]);	printf("\n");
	  }
	  if((no0=_TDJVSMDMssgNext(Md,0,msg,"VF8"	,32,vf8))>=0){
	    printf("VF8         [%2d]="		,no0);
	    for(i=0;i<no0;i++) printf("%e "	,vf8[i]);	printf("\n");
	  }

	}else if(sf	==0x0231){		// S2F49
	  if(_TDJVSMDMssgNext	 (Md,0,msg,"NOI1"	,64,vi4) > 0){
	    printf("NOI1        [%2d]\n"	,no1=vi4[0]);
	    for(i=0;i<no1;i++){
	      iname=String.format("NOI2:%d"		,i+1);
		// Refer to the comment in corresponding section of SubFuncsion.h
		// SubFuncsion.h ̊Yӏ̃RgQƂ邱
	      if(_TDJVSMDMssgNext (Md,0,msg,iname	,64,vi4)> 0){
		printf("%-12s[%2d]\n"		,iname,no2 =vi4[0]);
		for(j=0;j<no2;j++){
		  iname=String.format("WAFERID:%d:%d"	,i+1,j+1);
		  if((no0=_TDJVSMDMssgNext(Md,0,msg,iname,80,vs0))>=0){
		    printf("%-12s[%2d]='%s'\n",iname,no0  ,vs0[0]);
	  } } } } }

	}else if(sf	==0x060b){		// S6F11
	  if((no0=_TDJVSMDMssgNext(Md,0,msg,"DATAID"    ,128,vi2))>=0){
	    printf("DATAIDB     [%2d]="		,no0);
	    for(i=0;i<no0;i++) printf("%d "	,vi2[i]);	printf("\n");
	} } 
	_TDJVSMDMssgExit(Md,0,msg);
  } } }
}



// -----------------------------------------------------------------------------
// Display sent and received data ----------------------------------------------

static private int
DispData(
int		tp,		// i  : Message type
				//	=0 : Transmission result
				//	 1 : Received message
				//	 2 : Send message
byte		hd[],		// i  : SECS Message Header
int		did,		// i  : SECS Message Device ID
int		sf,		// i  : SECS Message SF-Code
int		xid,		// i  : SECS Message Transaction ID
byte		msg[],		// i  : SECS Message Body
int		len,		// i  : Byte length of 'msg'
int		rtn)		// i  : I/O return value
{
  int		ok=0,stat;

  if(rtn<0){							 ok=1;
    if((rtn<(-E_NOTCONNECT) || (-E_ILLBLOCK)<rtn) && rtn!=(-E_NODATA)){
      printf("ERROR  [%d]"			,rtn);
      if      (rtn==(-E_NODEV)){
	printf(" : No such device ID\n");	len=ok=0;
      }else if(rtn==(-E_2BIG )){
	printf(" : Data size to large\n");	len=ok=0;
      }else			printf("\n");
    }else{
      if(FuncType==0)	stat=_TDJVSCommStatus(Fd,0);
      else		stat=_TDJVSUDrvStatus(Fd,0);
      printf("STATUS = %d,%d : "		,-rtn,stat);
      switch(-rtn){
	case E_NODATA:		printf("No data\n");			break;
	case E_ILLBLOCK:	printf("Illegal block#\n");		break;
	case E_T1TIMEDOUT:	printf("T1 Timeout occur\n");		break;
	case E_T2TIMEDOUT:	printf("T2 Timeout occur\n");		break;
	case E_T3TIMEDOUT:	printf("T3 Timeout occur\n");		break;
	case E_T4TIMEDOUT:	printf("T4 Timeout occur\n");		break;
	case E_T5TIMEDOUT:	printf("T5 Timeout occur\n");		break;
	case E_T6TIMEDOUT:	printf("T6 Timeout occur\n");		break;
	case E_T7TIMEDOUT:	printf("T7 Timeout occur\n");		break;
	case E_T8TIMEDOUT:	printf("T8 Timeout occur\n");		break;
	case E_RETRYOVER:	printf("Retry over\n");			break;
	case E_CONNECT:		printf("Connected\n");			break;
	case E_SELECT:		printf("Selected   (0x%04x)\n",did);	break;
	case E_REJECT:		printf("Rejected XId=0x%04x\n",xid);	break;
	case E_DESELECT:	printf("Deselected (0x%04x)\n",did);	break;
	case E_SEPARATE:	printf("Separated  (0x%04x)\n",did);	break;
	case E_NOTCONNECT:	printf("Not connected\n");		break;
	default:		printf("\n");				break;
  } } }

  if(ok==0) DispSECSMssg(tp,hd,did,sf,xid,msg,len);

  return(0);
}



// =============================================================================
// S1F1 message construction and sending ---------------------------------------

static private int
SendS1F1()
{
  byte		hd[]=new byte[12],msg[]=new byte[16];
  int		rtn,sf=0x8101,cmd=0x0000,len=0;

  if(HeaderBO	!=0)	cmd=0x2000;
  if(FuncType	==0)	rtn=_TDJVSCommSend(Fd,cmd, -1,sf,  0,msg,len,hd);
  else			rtn=_TDJVSUDrvSend(Fd,cmd, -1,sf,  0,msg,len,hd);
  DispData(2,hd, -1,sf,rtn,msg,len,rtn);

  return(rtn);
}


// -----------------------------------------------------------------------------
// S1F2 message (Host) construction and sending --------------------------------

static private int
SendS1F2H(
int		did,
int		xid)
{
  byte		hd[]=new byte[12],msg[]=new byte[256];
  byte		vb[]=new byte[32];
  int		rtn,md,sf=0x0102,cmd=0x0000,len;
  
  if((MssgUseFile&0x80)==0){			// Do not use message definition
    md=		_TDJVSMssgInit   (   0,msg, 256,Fd);		// S1F2
		_TDJVSMssgBuild  (md,0,msg, 000,  0,vb);	// L0
    len=	_TDJVSMssgEnd    (md,0,msg);

  }else{					// Use message definition
		_TDJVSMDMssgInit (Md,0,msg, 256,"S1F2_H");
    len=	_TDJVSMDMssgEnd  (Md,0,msg);
  }

  if(len<0)		rtn=len;
  else{
    if(HeaderBO	!=0)	cmd=0x2000;
    if(FuncType	==0)	rtn=_TDJVSCommSend (Fd,cmd,did,sf,xid,msg,len,hd);
    else		rtn=_TDJVSUDrvSend (Fd,cmd,did,sf,xid,msg,len,hd);
  }
  DispData(2,hd,did,sf,rtn,msg,len,rtn);

  return(rtn);
}


// -----------------------------------------------------------------------------
// S1F2 message (Eqquipment) construction and sending --------------------------

static private int
SendS1F2E(
int		did,
int		xid)
{
  byte		hd[]=new byte[12],msg[]=new byte[256];
  byte		vb[]=new byte[32];
  int		rtn,md,sf=0x0102,cmd=0x0000,len;
  
  if((MssgUseFile&0x80)==0){			// Do not use message definition
    md=		_TDJVSMssgInit   (   0,msg, 256,Fd);		// S1F2
		_TDJVSMssgBuild  (md,0,msg, 000, 2,vb);		// L2
		_TDJVSMssgBuild  (md,0,msg, 020, 6,"EQUIP1");	//  MDLN
		_TDJVSMssgBuild  (md,0,msg, 020, 6,"01.000");	//  SOFTREV
    len=	_TDJVSMssgEnd    (md,0,msg);

  }else{					// Use message definition
		_TDJVSMDMssgInit (Md,0,msg, 256,"S1F2_E");
    len=	_TDJVSMDMssgEnd  (Md,0,msg);
  }

  if(len<0)		rtn=len;
  else{
    if(HeaderBO	!=0)	cmd=0x2000;
    if(FuncType	==0)	rtn=_TDJVSCommSend (Fd,cmd,did,sf,xid,msg,len,hd);
    else		rtn=_TDJVSUDrvSend (Fd,cmd,did,sf,xid,msg,len,hd);
  }
  DispData(2,hd,did,sf,rtn,msg,len,rtn);

  return(rtn);
}


// =============================================================================
// S1F21 message construction and sending --------------------------------------

static private int
SendS1F21()
{
  byte		hd []=new byte [12],msg[]=new byte[XMSGSIZE];
  byte		vb []={                      1,            (byte)0xff};
  byte		vt []={                      0,                     1};
  String	vs0  = "MSG31";
  String 	vs1  = " 31";
  String 	vs2  = "  (JVIo) ł";
  byte		vi1[]={                      2,                  -128};
  short		vi2[]={                  32767,                -32768};
  int		vi4[]={             0x7ffffffe,            0xfffffffe};
  long		vi8[]={    0x7fffffffffffffffL,   0x800080008000800bL};
  byte		vu1[]={                   0x7f,            (byte)0xff};
  short		vu2[]={                 0x7ffe,         (short)0xfffe};
  int		vu4[]={             0x7ffffffd,            0xfffffffd};
  long		vu8[]={    0x7fffffffffffffffL,   0x800080008000800bL};
  long		au8[]={  7, -6,  5, -4,  3, -2,  1,  0};
  float		vf4[]={  (float)9.87654321e-21,(float)-8.642097531e13};
  double	vf8[]={-1.9283746574839201e123,9.1827364546372819208e-213};
  String	fds;
  byte		fdt[]=new byte [301];
  byte		fd=0;
  int		rtn,md,sf=0x8100+21,cmd=0x0000,len=0,i;

  if((MssgUseFile&0x80)==0){			// Do not use message definition
    for(fd=0x20,i=0;i<300;fd++,i++){
      fdt[i]=fd;	if(fd==0x7e) fd=0x1f;
    } fdt[i]=0;
    fds=new String(fdt);
    md=		_TDJVSMssgInit   (   0,msg,XMSGSIZE,Fd);	// S1F21
  		_TDJVSMssgBuild  (md,0,msg, 000,  5,vb);	// L5
  		_TDJVSMssgBuild  (md,0,msg, 000,  2,vb);	//  L2
		_TDJVSMssgBuild  (md,0,msg, 010,  2,vb);	//   VB
		_TDJVSMssgBuild  (md,0,msg, 011,  2,vt);	//   VT
  		_TDJVSMssgBuild  (md,0,msg, 000,  3,vb);	//  L3
  		_TDJVSMssgBuild  (md,0,msg, 020,  6,vs0);	//   MDLN
  		_TDJVSMssgBuild  (md,0,msg, 021,  0,vs1);	//   SOFTREV
  		_TDJVSMssgBuild  (md,0,msg, 022,  0,vs2);	//   COMMENT
  		_TDJVSMssgBuild  (md,0,msg, 000,  9,vb);	//  L9
		_TDJVSMssgBuild  (md,0,msg, 031,  2,vi1);	//   VI1
		_TDJVSMssgBuild  (md,0,msg, 032,  2,vi2);	//   VI2
		_TDJVSMssgBuild  (md,0,msg, 034,  2,vi4);	//   VI4
		_TDJVSMssgBuild  (md,0,msg, 030,  2,vi8);	//   VI8
		_TDJVSMssgBuild  (md,0,msg, 051,  2,vu1);	//   VU1
		_TDJVSMssgBuild  (md,0,msg, 052,  2,vu2);	//   VU2
		_TDJVSMssgBuild  (md,0,msg, 054,  2,vu4);	//   VU4
		_TDJVSMssgBuild  (md,0,msg, 050,  2,vu8);	//   VU8
		_TDJVSMssgBuild  (md,0,msg, 050,  8,au8);	//   AU8
  		_TDJVSMssgBuild  (md,0,msg, 000,  2,vb);	//  L2
		_TDJVSMssgBuild  (md,0,msg, 044,  2,vf4);	//   VF4
		_TDJVSMssgBuild  (md,0,msg, 040,  2,vf8);	//   VF8
  		_TDJVSMssgBuild  (md,0,msg, 000,  3,vb);	//  L3
  		_TDJVSMssgBuild  (md,0,msg, 020,300,fds);	//   FILETEXT
		_TDJVSMssgBuild  (md,0,msg, 010,300,fdt);	//   FILEBIN
		_TDJVSMssgBuild  (md,0,msg, 052,150,fdt);	//   FILEU2
    len=	_TDJVSMssgEnd    (md,0,msg);

  }else{					// Use message definition
		_TDJVSMDMssgInit (Md,0,msg,XMSGSIZE,"S1F21_HE");
    len=	_TDJVSMDMssgEnd  (Md,0,msg);
  }

  if(len<0)		rtn=len;
  else{
    if(HeaderBO	!=0)	cmd=0x2000;
    if(FuncType	==0)	rtn=_TDJVSCommSend (Fd,cmd, -1,sf,  0,msg,len,hd);
    else		rtn=_TDJVSUDrvSend (Fd,cmd, -1,sf,  0,msg,len,hd);
  }
  DispData(2,hd, -1,sf,rtn,msg,len,rtn);

  return(rtn);
}


// -----------------------------------------------------------------------------
// S1F22 message (Host) construction and sending -------------------------------

static private int
SendS1F22(
int		did,
int		xid)
{
  byte		hd []=new byte [12],msg[]=new byte[XMSGSIZE];
  byte		vb []={                     2,              (byte)0xfe};
  byte		vt []={                     1,                       0};
  String	vs0  = "MSG32";
  String	vs1  = "޼ޮ";
  String	vs2  = "This is Rg (JVIo)";
  byte		vi1[]={                   127,                      -2};
  short		vi2[]={                 32766,                  -32768};
  int		vi4[]={            2147483646,             -2147483648};
  long		vi8[]={   0x8000000000000000L,     0x7ffffffffffffffeL};
  byte		vu1[]={              (byte)254,             (byte)0xff};
  short		vu2[]={           (short)65534,          (short)0xffff};
  int		vu4[]={       (int)4294967294L,        (int)0xffffffff};
//long		vu8[]={  18446744073709551614L,    0xffffffffffffffffL};
  long		vu8[]={    0xffffffffffffffffL,    0xffffffffffffffffL};
  long		au8[]={ -1,  2, -3,  4, -5,  6, -7,  0};
  float		vf4[]={  (float)7.89012345e-12,(float)-4.321098765e31};
  double	vf8[]={ 5.6473829101928374e189,-3.2109876543210987654e-179};
  int		rtn,md,sf=0x0100+22,cmd=0x0000,len=0;
  
  if((MssgUseFile&0x80)==0){			// Do not use message definition
    md=		_TDJVSMssgInit   (   0,msg,XMSGSIZE,Fd);	// S1F22
  		_TDJVSMssgBuild  (md,0,msg, 000,  4,vb);	// L4
  		_TDJVSMssgBuild  (md,0,msg, 000,  2,vb);	//  L2
		_TDJVSMssgBuild  (md,0,msg, 010,  2,vb);	//   VB
		_TDJVSMssgBuild  (md,0,msg, 011,  2,vt);	//   VT
  		_TDJVSMssgBuild  (md,0,msg, 000,  3,vb);	//  L3
  		_TDJVSMssgBuild  (md,0,msg, 020,  6,vs0);	//   MDLN
  		_TDJVSMssgBuild  (md,0,msg, 021,  0,vs1);	//   SOFTREV
  		_TDJVSMssgBuild  (md,0,msg, 022,  0,vs2);	//   COMMENT
  		_TDJVSMssgBuild  (md,0,msg, 000,  9,vb);	//  L9
		_TDJVSMssgBuild  (md,0,msg, 031,  2,vi1);	//   VI1
		_TDJVSMssgBuild  (md,0,msg, 032,  2,vi2);	//   VI2
		_TDJVSMssgBuild  (md,0,msg, 034,  2,vi4);	//   VI4
		_TDJVSMssgBuild  (md,0,msg, 030,  2,vi8);	//   VI8
		_TDJVSMssgBuild  (md,0,msg, 051,  2,vu1);	//   VU1
		_TDJVSMssgBuild  (md,0,msg, 052,  2,vu2);	//   VU2
		_TDJVSMssgBuild  (md,0,msg, 054,  2,vu4);	//   VU4
		_TDJVSMssgBuild  (md,0,msg, 050,  2,vu8);	//   VU8
		_TDJVSMssgBuild  (md,0,msg, 050,  8,au8);	//   AU8
  		_TDJVSMssgBuild  (md,0,msg, 000,  2,vb);	//  L2
		_TDJVSMssgBuild  (md,0,msg, 044,  2,vf4);	//   VF4
		_TDJVSMssgBuild  (md,0,msg, 040,  2,vf8);	//   VF8
    len=	_TDJVSMssgEnd    (md,0,msg);

  }else{					// Use message definition
		_TDJVSMDMssgInit (Md,0,msg,XMSGSIZE,"S1F22_HE");
    len=	_TDJVSMDMssgEnd  (Md,0,msg);
  }

  if(len<0)		rtn=len;
  else{
    if(HeaderBO	!=0)	cmd=0x2000;
    if(FuncType ==0)	rtn=_TDJVSCommSend (Fd,cmd,did,sf,xid,msg,len,hd);
    else		rtn=_TDJVSUDrvSend (Fd,cmd,did,sf,xid,msg,len,hd);
  }
  DispData(2,hd,did,sf,rtn,msg,len,rtn);

  return(rtn);
}


// =============================================================================
// S2F49 message construction and sending --------------------------------------

static private int
SendS2F49()
{
  byte		hd[]=new byte[12],msg[]=new byte[XMSGSIZE];
  byte		vb[]=new byte[32];
  int		vi[]=new int[4];
  String	str,itm;
  int		rtn,md,sf=0x8200+49,cmd=0x0000,no1,no2,len,i,j;

  cnt249++;	no1=(cnt249%2)+1;	no2=(cnt249%10)+1;
  str=String.format("LOTID (%04d)"	,cnt249);
  
  if((MssgUseFile&0x80)==0){			// Do not use message definition
    md=		_TDJVSMssgInit   (   0,msg,XMSGSIZE,Fd);	// S2F49
  		_TDJVSMssgBuild  (md,0,msg, 000,  3,vb);	// L3
    vb[0]=0;	_TDJVSMssgBuild  (md,0,msg, 010,  1,vb);	//  DATAIDB
  		_TDJVSMssgBuild  (md,0,msg, 020,  0,"LOAD");	//  RCMD
  		_TDJVSMssgBuild  (md,0,msg, 000,  4,vb);	//  L4
    vb[0]=1;	_TDJVSMssgBuild  (md,0,msg, 010,  1,vb);	//   STID
    vb[0]=0;	_TDJVSMssgBuild  (md,0,msg, 010,  1,vb);	//   MTKD
  		_TDJVSMssgBuild  (md,0,msg, 020, 20,str);	//   LOTID
		_TDJVSMssgBuild  (md,0,msg, 000,no1,vb);	//   L[no1]
    for(i=0;i<no1;i++){
		_TDJVSMssgBuild  (md,0,msg, 000,no2,vb);	//    L[no2]
      for(j=0;j<no2;j++){
		_TDJVSMssgBuild  (md,0,msg, 000,  2,vb);	//     L[2]
	str=String.format("WAFER(%04d-%d-%02d)",cnt249,i+1,i+2);
  		_TDJVSMssgBuild  (md,0,msg, 020, 20,str);	//      WAFERID
	str=String.format("PPID (%04d-%d-%02d)",cnt249,i+1,i+2);
  		_TDJVSMssgBuild  (md,0,msg, 020, 16,str);	//      PPID
    } }
    len=	_TDJVSMssgEnd    (md,0,msg);

  }else{					// Use message definition
		_TDJVSMDMssgInit (Md,0,msg,XMSGSIZE,"S2F49_H");
  		_TDJVSMDMssgBuild(Md,0,msg,"LOTID",0,str);	//   LOTID
    vi[0]=no1;	_TDJVSMDMssgBuild(Md,0,msg,"NOI1" ,1,vi);	//   L[no1]
    for(i=0;i<no1;i++){
      itm  =String.format("NOI2:%d"			,i+1);
      vi[0]=no2;_TDJVSMDMssgBuild(Md,0,msg,itm   ,1,vi);	//    L[no2]
      for(j=0;j<no2;j++){
	itm=String.format("WAFERID:%d:%d"		,i+1,j+1);
	str=String.format("WAFERID[%04d-%d-%02d]",cnt249,i+1,j+1);
  		_TDJVSMDMssgBuild(Md,0,msg,itm    ,0,str);	//     WAFERID
	itm=String.format("PPID:%d:%d"			,i+1,j+1);
	str=String.format("PPID [%04d-%d-%02d]"	 ,cnt249,i+1,j+1);
  		_TDJVSMDMssgBuild(Md,0,msg,itm    ,0,str);	//     PPID
    } }
    len=	_TDJVSMDMssgEnd  (Md,0,msg);
  }

  if(len<0)		rtn=len;
  else{
    if(HeaderBO	!=0)	cmd=0x2000;
    if(FuncType	==0)	rtn=_TDJVSCommSend (Fd,cmd, -1,sf,  0,msg,len,hd);
    else		rtn=_TDJVSUDrvSend (Fd,cmd, -1,sf,  0,msg,len,hd);
  }
  DispData(2,hd, -1,sf,rtn,msg,len,rtn);

  return(rtn);
}


// -----------------------------------------------------------------------------
// S2F50 message construction and sending --------------------------------------

static private int
SendS2F50(
int		did,
int		xid)
{
  byte		hd[]=new byte[12],msg[]=new byte[256];
  byte		vb[]=new byte[32];
  String	str;
  int		rtn,md,sf=0x0200+50,cmd=0x0000,len;
  
  str=String.format("LOTID (%04d)"		,++cnt250);

  if((MssgUseFile&0x80)==0){			// Do not use message definition
    md=		_TDJVSMssgInit   (   0,msg, 256,Fd);		// S2F50
		_TDJVSMssgBuild  (md,0,msg, 000,  2,vb);	// L2
    vb[0]=0;	_TDJVSMssgBuild  (md,0,msg, 010,  1,vb);	//  HCACK
		_TDJVSMssgBuild  (md,0,msg, 000,  2,vb);	//  L2
  		_TDJVSMssgBuild  (md,0,msg, 020, 16,"PPID");	//   PPID
  		_TDJVSMssgBuild  (md,0,msg, 020, 20,str);	//   LOTID
    len=	_TDJVSMssgEnd    (md,0,msg);

  }else{					// Use message definition
		_TDJVSMDMssgInit (Md,0,msg, 256,"S2F50_E");
		_TDJVSMDMssgBuild(Md,0,msg,"LOTID",  0,str);	//   LOTID
    len=	_TDJVSMDMssgEnd  (Md,0,msg);
  }

  if(len<0)		rtn=len;
  else{
    if(HeaderBO	!=0)	cmd=0x2000;
    if(FuncType	==0)	rtn=_TDJVSCommSend (Fd,cmd,did,sf,xid,msg,len,hd);
    else		rtn=_TDJVSUDrvSend (Fd,cmd,did,sf,xid,msg,len,hd);
  }
  DispData(2,hd,did,sf,rtn,msg,len,rtn);

  return(rtn);
}


// =============================================================================
// S6F11 message construction and sending --------------------------------------

static private int
SendS6F11()
{
  byte		hd[]=new byte[12],msg[]=new byte[256];
  short		vs[]=new short[4];
  int		rtn,md,sf=0x8600+11,cmd=0x0000,len;

  if((++cnt611)==32768) cnt611=0;

  if((MssgUseFile&0x80)==0){			// Do not use message definition
    md=		_TDJVSMssgInit   (   0,msg, 256,Fd);		// S6F11
  		_TDJVSMssgBuild  (md,0,msg, 000,  3,vs);	// L3
    vs[0]=(short)cnt611;
		_TDJVSMssgBuild	 (md,0,msg, 052,  1,vs);	//  DATAID
    vs[0]=8;	_TDJVSMssgBuild  (md,0,msg, 052,  1,vs);	//  CEID
  		_TDJVSMssgBuild  (md,0,msg, 000,  3,vs);	//  L3
  		_TDJVSMssgBuild  (md,0,msg, 020, 16,"DATA1");	//   DATA1
  		_TDJVSMssgBuild  (md,0,msg, 020, 16,"DATA2");	//   DATA2
  		_TDJVSMssgBuild  (md,0,msg, 020, 14,"YYYYMMDDhhmmss");
			// @TIME Actually I set current time, but omitted.
			// @TIME {͌ݎݒ肷̂Aȗ
    len=	_TDJVSMssgEnd    (md,0,msg);

  }else{					// Use message definition
		_TDJVSMDMssgInit (Md,0,msg, 256,"S6F11_E0");
    vs[0]=(short)cnt611;
		_TDJVSMDMssgBuild(Md,0,msg,"DATAID",1,vs);	//  DATAID
    len=	_TDJVSMDMssgEnd  (Md,0,msg);
  }

  if(len<0)		rtn=len;
  else{
    if(HeaderBO	!=0)	cmd=0x2000;
    if(FuncType	==0)	rtn=_TDJVSCommSend (Fd,cmd, -1,sf,  0,msg,len,hd);
    else		rtn=_TDJVSUDrvSend (Fd,cmd, -1,sf,  0,msg,len,hd);
  }
  DispData(2,hd, -1,sf,rtn,msg,len,rtn);

  return(rtn);
}



// -----------------------------------------------------------------------------
// S6F12 message construction and sending --------------------------------------

static private int
SendS6F12(
int		did,
int		xid)
{
  byte		hd[]=new byte[12],msg[]=new byte[256];
  byte		vb[]=new byte[32];
  int		rtn,md,sf=0x0600+12,cmd=0x0000,len;

  if((MssgUseFile&0x80)==0){			// Do not use message definition
    md=		_TDJVSMssgInit	(   0,msg, 256,Fd);		// S6F12
    vb[0]=0;	_TDJVSMssgBuild	(md,0,msg, 010,  1,vb);		// ACKC
    len=	_TDJVSMssgEnd 	(md,0,msg);

  }else{					// Use message definition
		_TDJVSMDMssgInit(Md,0,msg, 256,"S6F12_H");
    len=	_TDJVSMDMssgEnd (Md,0,msg);
  }

  if(len<0)		rtn=len;
  else{
    if(HeaderBO	!=0)	cmd=0x2000;
    if(FuncType	==0)	rtn=_TDJVSCommSend (Fd,cmd,did,sf,xid,msg,len,hd);
    else		rtn=_TDJVSUDrvSend (Fd,cmd,did,sf,xid,msg,len,hd);
  }
  DispData(2,hd,did,sf,rtn,msg,len,rtn);

  return(rtn);
}



// =============================================================================
// Callback function ===========================================================

static private int
CBRecvProc(
int		req,		// i  : Request code to library
int		rtn,		// i  : Return code from library
int		did,		// i  : SECS Message Device ID
int		xsf,		// i  : SECS Message SF-Code
int		xid,		// i  : SECS Message Transaction ID
byte		xhd[],		// i  : SECS Message Header
byte		xmsg[])		// i  : SECS Message Body
{
  StringBuffer	rname=new StringBuffer(),sname=new StringBuffer();
  byte		hd[]=new byte[12],msg[]=new byte[XMSGSIZE];
  int		sf[]=new int[1];
  int		len,cmd=0x0000;

  DispData(1,xhd,did,xsf,xid,xmsg,rtn,rtn);

  if      (req== 0 && rtn>=0){
    if((MssgUseFile&0x80)==0){// If you do not use message definition file,
				// check SF-Code on your own, determine the
				// necessity of sending secondary message, and
				// send it if necessary.
				// bZ[W`t@CgpȂꍇ́A
	    			// ͂ SF-Code 𒲂ׁAQbZ[W̑o
				// Kv𔻒fAKvȏꍇ́AoB
      if     (xsf==(0x8100+ 1)&&OType==0)	SendS1F2H (did,xid);
      else if(xsf==(0x8100+ 1)&&OType==1)	SendS1F2E (did,xid);
      else if(xsf==(0x8100+21))			SendS1F22 (did,xid);
      else if(xsf==(0x8200+49))			SendS2F50 (did,xid);
      else if(xsf==(0x8600+11))			SendS6F12 (did,xid);

    }else if((xsf & 0x8000)!=0){// When using a message definition file, this
				// sample uses automatic reply function.
				// bZ[W`t@Cgpꍇ́A{T
				// vł́AԐM@\gpB
      msg=xmsg;
      if((len=_TDJVSMDMssgAutoRes(Md,0,xhd
			,msg,rtn,XMSGSIZE,rname,sname,sf))	>=0){
	printf("RECV %s ..  Auto respond %s [S%dF%d]\n"
		,rname.toString(),sname.toString(),sf[0]/0x100,sf[0]&0xff);
	if(HeaderBO!=0)	cmd=0x2000;
	if(FuncType==0)	rtn=_TDJVSCommSend(Fd,cmd,did,sf[0],xid,msg,len,hd);
	else		rtn=_TDJVSUDrvSend(Fd,cmd,did,sf[0],xid,msg,len,hd);
	DispData(2,hd,did,sf[0],xid,msg,len,rtn);
      }else{
	if(len!=(-930) && len!=(-931)){
	  printf("RECV Auto response error (%d)\n",len);
  } } } }

  return(0);
}



// -----------------------------------------------------------------------------

static class RecvProcThread extends Thread{
public void
run()
{
  byte		msg[]=new byte[XMSGSIZE],hd[]=new byte[12];
  int		xid[]=new int[1],did[]=new int[1],sf[]=new int[1];
  int		rtn,cmd,req;

  for(;Break==0;){
    req=0;
    if(HeaderBO	!=0)	cmd=0x2000;
    else		cmd=0x0000;
    if(FuncType	==0)	rtn=_TDJVSCommRecv(Fd,cmd,did,sf,xid,msg,XMSGSIZE,hd);
    else		rtn=_TDJVSUDrvRecv(Fd,cmd,did,sf,xid,msg,XMSGSIZE,hd);
    if(rtn==(-951)){
      Sleep(100);
    }else{
      if((-1000)<rtn && rtn<(-959))	req=(-rtn)-900;
      CBRecvProc(req,rtn,did[0],sf[0],xid[0],hd,msg);
  } }
}
}



// =============================================================================
// Main process ----------------------------------------------------------------

public static void
main(
String[]		argv)
{
  int			opt=0;

  // ---------------------------------------------------------------------------
  {				// Display TDS information
    int			bits;	// This code has nothing to do with running TDS.
    StringBuffer	tvers=new StringBuffer(),svers=new StringBuffer();
    StringBuffer	tdate=new StringBuffer(),sdate=new StringBuffer();
    StringBuffer	tcomp=new StringBuffer(),scomp=new StringBuffer();
    bits=_TDJVSGetVersion(tvers,svers,tdate,sdate,tcomp,scomp);
    printf("Version = %-20s %-20s\n"	,tvers.toString(),svers.toString());
    printf("Date    = %-20s %-20s\n"	,tdate.toString(),sdate.toString());
    printf("Compile = %-20s %-20s\n"	,tcomp.toString(),scomp.toString());
    printf("#of Bit = %d\n"		,bits);
  }
  // ---------------------------------------------------------------------------

  // ---------------------------------------------------------------------------
  if(FuncType		!=0)		opt|=0x0001;
  if(HeaderBO		!=0)		opt|=0x0002;
  if(MssgDispType	!=0x00)		opt|=0x0004;
  if(MssgUseNextL	!=0)		opt|=0x0008;
  if(MssgUseFile	!=0x00)		opt|=0x0010;
  if(UseRecvThrd	!=0)		opt|=0x0020;
  if(SyncStatus		!=0)		opt|=0x8000;
  if     (argv.length>1){	// Analysis of running parameters
    if	    (argv[1].length()	>1			   &&
	    (argv[1].substring(0,1).equalsIgnoreCase("x"))){
      opt=Integer.parseInt(argv[1].substring(1),16);
    }else if(argv[1].length()	>2			   &&
	    (argv[1].substring(0,1).equals	    ("0")) &&
	    (argv[1].substring(1,2).equalsIgnoreCase("x"))){
      opt=Integer.parseInt(argv[1].substring(2),16);
    }else{
      opt=Integer.parseInt(argv[1]);
    }
    if((opt&0x0001)!=0) FuncType    =1;		else FuncType    =0;
    if((opt&0x0002)!=0) HeaderBO    =1;		else HeaderBO	 =0;
    if((opt&0x0004)!=0) MssgDispType=0x20;	else MssgDispType=0x00;
    if((opt&0x0008)!=0) MssgUseNextL=1;		else MssgUseNextL=0;
    if((opt&0x0010)!=0) MssgUseFile =0x80;	else MssgUseFile =0x00;
    if((opt&0x0020)!=0) UseRecvThrd =1;		else UseRecvThrd =0;
    if((opt&0x8000)!=0) SyncStatus  =1;		else SyncStatus	 =0;
  }
  if(argv.length>2)	SendWait    =Integer.parseInt(argv[2]);
  if(argv.length>3)	SendSleep   =Integer.parseInt(argv[3]);

  printf("option      =0x%02x,%d\n"		,opt,opt);
  printf("FuncType    =%d\n"			,FuncType);
  printf("HeaderBO    =%d\n"			,HeaderBO);
  printf("MssgDispType=0x%02x\n"		,MssgDispType);
  printf("MssgUseNextL=%d\n"			,MssgUseNextL);
  printf("MssgUseFile =0x%02x\n"		,MssgUseFile);
  printf("UseRecvThrd =%d\n"			,UseRecvThrd);
  printf("SyncStatus  =%d\n"			,SyncStatus);
  printf("SendWait    =%d\n"			,SendWait);
  printf("SendSleep   =%d\n"			,SendSleep);
  // ---------------------------------------------------------------------------

  if     (argv.length<1)			Usage();
  else if(argv[0].equals("h")==false &&
	  argv[0].equals("e")==false)		Usage();
  else{
    if(argv[0].equals("h")   ==true)		Host ();
    else					Equip();
  }
}


// -----------------------------------------------------------------------------

static private void
Usage()
{
  printf("\nUsage: JVIo {h|e} [option [send_wait [send_sleep]]]\n");
  printf("\n  option\n");
  printf(  "   F           54 3210\n");
  printf(  "  +----+----+----+----+\n");
  printf(  "   |           || |||+-- Function to use\n");
  printf(  "   |           || |||    =0: _TDSCommXxxxx()           1:_TDSUDrvXxxx()\n");
  printf(  "   |           || ||+--- SECS Header Byte order\n");
  printf(  "   |           || ||     =0: System order              1: Network order\n");
  printf(  "   |           || |+---- Format for displaying messages in SECS list format\n");
  printf(  "   |           || |      =0: TDS Format                1: SML Format\n");
  printf(  "   |           || +----- Whether to use MssgNextL() for SECS message display\n");
  printf(  "   |           ||        =0: Not use                   1: Use\n");
  printf(  "   |           |+------- Whether to use the SECS message definition file\n");
  printf(  "   |           |         =0: Not use                   1: Use\n");
  printf(  "   |           +-------- Whether to execute SECS message reception in a dedicated thread\n");
  printf(  "   |                     =0: No                        1: Yes\n");
  printf(  "   +-------------------- Synchronize connection status\n");
  printf(  "                         =0: No                        1: Yes\n");
  printf(  "  send_wait   : Message sending interval     (%5dms)\n",SendWait);
  printf(  "  send_sleep  : Wait time                    (%5dms)\n",SendSleep);
  printf("\n");
}



// =============================================================================
// Host side process -----------------------------------------------------------

static private void
Host()
{
  BufferedReader	in;
  RecvProcThread	th;
  String		str;
  byte			hd[]=new byte[12],msg[]=new byte[XMSGSIZE];
  int			xid[]=new int[1],did[]=new int[1],sf[]=new int[1];
  int			req,rtn,omd,cmd=0,mno,dsd=0,xsd=0,stat,i;

  OType=0;		Break=0;
  in=new BufferedReader(new InputStreamReader(System.in));

L0:{
  if(SyncStatus	==0)	omd=0x0002;
  else			omd=0x1002;
  if(FuncType	==0)	Fd=_TDJVSCommOpen(omd,PARAMFILE,"HOST");
  else			Fd=_TDJVSUDrvOpen(omd,PARAMFILE,"HOST",UDRV_MASK);
  if(Fd							  < 0) break L0;
  printf("(H) Opened (%d)\n",Fd);
  if((MssgUseFile&0x80)!=0) Md=_TDJVSMDMssgInitialize(0x4000,Fd,"");

  if(UseRecvThrd!=0){
    th=new RecvProcThread();
    th.start();
  }

  if(SECS_MODE!=0){
	// In case of HSMS, wait for Select state.
	// For SECS-1 connection, the following connection status confirmation
	// processing may be performed.
	// (Note 1) If connection status is not synchronized with (omd&0x1000)==0
	// at CommOpen() or UDrvOpen(), connection status change report must be
	// received with CommRecv() or UDrvRecv().  (The processing is not
	// described here.)
	// If you are using a receive thread with UseRecvThrd!=0, you don't need,
	// because that thread will receive connection state change reports.
	// (Note 2) In the first place, even if it is not in Selected state, if
	// you think that "_TDSCommSend() etc. will only result in an error
	// (E_ILSEQ)", at least in this AP, this "waiting for Selected state"
	// operation is not necessary.
	//
	// HSMS ڑ̏ꍇASelect ԂɂȂ̂҂B
	// SECS-1 ڑ̏ꍇȉ̐ڑXe[^XmF͍sĂ܂ȂB
	// (1) CommOpen()AUDrvOpen()  (omd&0x1000)==0 ƂĐڑXe[^X
	// ̓sȂꍇ́ACommRecv()AUDrvRecv() ŁAڑԕω񍐂
	// MȂ΂ȂȂB
	// UseRecvThrd!=0 ŎMXbhgpĂꍇ́ÃXbhɂ
	// ڑԕω񍐂M̂ŁA̕Kv͂ȂB
	// (2) A Selected ԂɂȂĂȂĂAuȍ~
	// _TDSCommSend() G[ (E_ILSEQ) ɂȂ邾vƍl΁AȂƂ
	//  AP ł́ÁuSelected ԂɂȂ̂҂vƂ쎩̂Kv
	// ȂB
    for(;;){
      if(SyncStatus==0 && UseRecvThrd==0){
	if(FuncType==0)	_TDJVSCommRecv(Fd,0x0000,did,sf,xid,msg,XMSGSIZE,hd);
	else		_TDJVSUDrvRecv(Fd,0x0000,did,sf,xid,msg,XMSGSIZE,hd);
      }
      if(FuncType==0)	rtn=_TDJVSCommStatus(Fd,0x0000);
      else		rtn=_TDJVSUDrvStatus(Fd,0x0000);
      if(rtn							< 0) break L0;
      if(rtn							==3) break;
      Sleep(100);
    }
    printf("(H) Selected\n");
  }

// (Note 3) In other words, when SyncStatus!=0, the following code is OK for both
//    SECS-1 and HSMS.
// (3)  SyncStatus!=0 ̏ꍇASECS-1AHSMS Ɉȉ̃R[hłnjƂ
//for(;;){
//  if(FuncType==0)	rtn=_TDJVSCommStatus(Fd,0x0000)
//  else		rtn=_TDJVSUDrvStatus(Fd,0x0000)
//  if(rtn							< 0) break L0;
//  if(rtn							==3) break;
//  Sleep(100);
//}

  for(;;){		rtn=0;
    if(UseRecvThrd==0)	str="Req (0:Exit 1:Recv 2:Send 4:Statistic Report) : ";
    else		str="Req (0:Exit 2:Send 3:Continuous 4:Statistic Report) : ";
    req=ReadInt(in,str,10);
    if      (req==0){						break;

    }else if(req==1){
    //if(HeaderBO!=0)	cmd=0x2000;
    //else		cmd=0x0000;
      if(FuncType==0)	rtn=_TDJVSCommRecv(Fd,cmd,did,sf,xid,msg,XMSGSIZE,hd);
      else		rtn=_TDJVSUDrvRecv(Fd,cmd,did,sf,xid,msg,XMSGSIZE,hd);
      if(HeaderBO!=0){	// The mode setting of CommRecv() is sufficient, but
			// for testing purposes, CommHeaderBO() is used here.
      			// CommRecv()  mode ݒłƂ͑邪Ał
			// ̂߁A CommHeaderBO() gpB
	if(FuncType==0)	_TDJVSCommHeaderBO(Fd,0x00,hd);
	else		_TDJVSUDrvHeaderBO(Fd,0x00,hd);
      }
      if(rtn	>=0){	dsd=did[0];	xsd=xid[0];}
      DispData(1,hd,did[0],sf[0],xid[0],msg,rtn,rtn);

    }else if(req==2){
      if(UseRecvThrd==0){
	str="Message(1:S1F1 2:S1F21 3:S2F49   6:S1F2 7:S1F22 8:S6F12) : ";
      }else{
	str="Message(1:S1F1 2:S1F21 3:S2F49) : ";
      }
      mno=ReadInt(in,str,10);
      switch(mno){
	case 1: rtn=SendS1F1 ();		break;
	case 2: rtn=SendS1F21();		break;
	case 3: rtn=SendS2F49();		break;
	case 6: rtn=SendS1F2H(dsd,xsd);		break;
	case 7: rtn=SendS1F22(dsd,xsd);		break;
	case 8: rtn=SendS6F12(dsd,xsd);		break;
      }

    }else if(req==3){
      for(Break=i=0;Break==0;i++){
	Sleep(SendWait);
	if(FuncType==0)	stat=_TDJVSCommStatus(Fd,0);
	else		stat=_TDJVSUDrvStatus(Fd,0);
	if(stat==3){
	  if((rtn=SendS2F49())	< 0){
	    printf("(H) Send error (%d)\n",rtn);	Sleep(ERROR_SLEEP);
	  }
	  if((i%SEND_NO)			==0)	Sleep(SendSleep);
      } }
      rtn=0;

    }else if(req==4){
      int		sp[]=new int[16];
      int		noe,nort,ii,s,f,d;
      for(noe=ii=0;ii<2;ii++){
	if(ii==0) printf("\n[Sent Primary Message Table]\n");
	else	  printf("\n[Received Primary Message Table]\n");
	printf( "   SC  FC: Primary: Second.:Async T3TO"+
		": S9F1   F3   F5   F7   F9  F11  F13:  Total.ms (   Ave. ms)\n");
	for(s=0;s<128;s++){	for(f=0;f<256;f++){
	  if(FuncType	==0)	rtn=_TDJVSCommSInfoGet(Fd,ii,(s<<8)|f,sp);
	  else			rtn=_TDJVSUDrvSInfoGet(Fd,ii,(s<<8)|f,sp);
	  if(rtn	==0){	noe++;
	    if((d=f)	!=0)	f++;
	    if((nort=sp[1])==0) nort=1;
	    printf("  %3d %3d:%8d:%8d:%5d%5d"+
		":%5d%5d%5d%5d%5d%5d%5d:%8dms (%8.1fms)\n"
		,s,d	,sp[ 0],sp[ 1],sp[ 2],sp[ 3]
			,sp[ 4],sp[ 5],sp[ 6],sp[ 7] ,sp[ 8]
			,sp[ 9],sp[10],sp[14],(double)sp[14]/(double)nort);
	  }else{
	    if(f	!=0)	f++;
      } }}}
      if(noe==0){
	printf("\nSet STATISINFO=1 in the .ini file when using this function.\n\n");
      }else	printf("\n");
      rtn=0;
    }

    if(rtn<(-999) || ((-900)<rtn && rtn<0)){
      printf("(H) I/O Error (%d)\n"	,rtn);
  } }
  }	// L0:

  Break= 1;
  if(Fd> 0){
    if(FuncType==0)	_TDJVSCommClose(Fd,0);
    else		_TDJVSUDrvClose(Fd,0);
  } else		printf("(H) Error (%d)\n"	,Fd);
  if(Md> 0)		_TDJVSMDMssgTerminate(Md,0);
}



// =============================================================================
// Equipment side process ------------------------------------------------------

static private void
Equip()
{
  BufferedReader	in;
  RecvProcThread	th;
  String		str;
  byte			hd[]=new byte[12],msg[]=new byte[XMSGSIZE];
  int			xid[]=new int[1],did[]=new int[1],sf[]=new int[1];
  int			req,rtn,omd,cmd=0,mno,dsd=0,xsd=0,stat,i;

  OType=1;		Break=0;
  in=new BufferedReader(new InputStreamReader(System.in));

L0:{
  if(SyncStatus	==0)	omd=0x0002;
  else			omd=0x1002;
  if(FuncType	==0)	Fd=_TDJVSCommOpen(omd,PARAMFILE,"EQUIP");
  else			Fd=_TDJVSUDrvOpen(omd,PARAMFILE,"EQUIP",UDRV_MASK);
  if(Fd								  < 0) break L0;
  printf("(E) Opened (%d)\n",Fd);
  if((MssgUseFile&0x80)!=0) Md=_TDJVSMDMssgInitialize(0x4000,Fd,"");

  if(UseRecvThrd!=0){
    th=new RecvProcThread();
    th.start();
  }

  if(SECS_MODE!=0 && FuncType==0){ // In case of HSMS and use TDSCommXxxxx()
				    // HSMS  TDSCommXxxx() gp̏ꍇ
    if(_TDJVSCommSend(Fd,0x0100,0,0,0,msg,0,hd)			< 0) break L0;
    printf("(E) Connected\n");
    if(_TDJVSCommSend(Fd,0x0200,0,0,0,msg,0,hd)			< 0) break L0;
    printf("(E) Select request\n");

    for(;;){	// Wait for Select state. See the description for Host().
    		// Select ԂɂȂ̂҂B Host() ł̐QƂ邱ƁB
      if(SyncStatus==0 && UseRecvThrd==0){
	if(FuncType==0)	_TDJVSCommRecv(Fd,0x0000,did,sf,xid,msg,XMSGSIZE,hd);
	else		_TDJVSUDrvRecv(Fd,0x0000,did,sf,xid,msg,XMSGSIZE,hd);
      }
      if(FuncType==0)	rtn=_TDJVSCommStatus(Fd,0x0000);
      else		rtn=_TDJVSUDrvStatus(Fd,0x0000);
      if(rtn							< 0) break L0;
      if(rtn							==3) break;
      Sleep(100);
    }
    printf("(E) Selected\n");
  }

  for(;;){		rtn=0;
    if(UseRecvThrd==0)	str="Req (0:Exit 1:Recv 2:Send 4:Statistic Report) : ";
    else		str="Req (0:Exit 2:Send 3:Continuous 4:Statistic Report) : ";
    req=ReadInt(in,str,10);
    if      (req==0){						     break;

    }else if(req==1){
    //if(HeaderBO!=0)	cmd=0x2000;
    //else		cmd=0x0000;
      if(FuncType==0)	rtn=_TDJVSCommRecv(Fd,cmd,did,sf,xid,msg,XMSGSIZE,hd);
      else		rtn=_TDJVSUDrvRecv(Fd,cmd,did,sf,xid,msg,XMSGSIZE,hd);
      if(HeaderBO!=0){	// The mode setting of CommRecv() is sufficient, but
			// for testing purposes, CommHeaderBO() is used here.
      			// CommRecv()  mode ݒłƂ͑邪Ał
			// ̂߁A CommHeaderBO() gpB
	if(FuncType==0)	_TDJVSCommHeaderBO(Fd,0x00,hd);
	else		_TDJVSUDrvHeaderBO(Fd,0x00,hd);
      }
      if(rtn	>=0){	dsd=did[0];	xsd=xid[0];}
      if(rtn	>=0){	dsd=did[0];	xsd=xid[0];}
      DispData(1,hd,did[0],sf[0],xid[0],msg,rtn,rtn);

    }else if(req==2){
      if(UseRecvThrd==0){
	str="Message(1:S1F1 2:S1F21 3:S6F11   6:S1F2 7:S1F22 8:S2F50) : ";
      }else{
	str="Message(1:S1F1 2:S1F21 3:S6F11) : ";
      }
      mno=ReadInt(in,str,10);
      switch(mno){
	case 1: rtn=SendS1F1 ();		break;
	case 2: rtn=SendS1F21();		break;
	case 3: rtn=SendS6F11();		break;
	case 6: rtn=SendS1F2E(dsd,xsd);		break;
	case 7: rtn=SendS1F22(dsd,xsd);		break;
	case 8: rtn=SendS2F50(dsd,xsd);		break;
      }

    }else if(req==3){
      for(Break=i=0;Break==0;i++){
	Sleep(SendWait);
	if(FuncType==0)	stat=_TDJVSCommStatus(Fd,0);
	else		stat=_TDJVSUDrvStatus(Fd,0);
	if(stat==3){
	  if((rtn=SendS6F11())	< 0){
	    printf("(E) Send error (%d)\n",rtn);	Sleep(ERROR_SLEEP);
	  }
	  if((i%SEND_NO)			==0)	Sleep(SendSleep);
      } }
      rtn=0;

    }else if(req==4){
      int		sp[]=new int[16];
      int		noe,nort,ii,s,f,d;
      for(noe=ii=0;ii<2;ii++){
	if(ii==0) printf("\n[Sent Primary Message Table]\n");
	else	  printf("\n[Received Primary Message Table]\n");
	printf( "   SC  FC: Primary: Second.:Async T3TO"+
		": S9F1   F3   F5   F7   F9  F11  F13:  Total.ms (   Ave. ms)\n");
	for(s=0;s<128;s++){	for(f=0;f<256;f++){
	  if(FuncType	==0)	rtn=_TDJVSCommSInfoGet(Fd,ii,(s<<8)|f,sp);
	  else			rtn=_TDJVSUDrvSInfoGet(Fd,ii,(s<<8)|f,sp);
	  if(rtn	==0){	noe++;
	    if((d=f)	!=0)	f++;
	    if((nort=sp[1])==0) nort=1;
	    printf("  %3d %3d:%8d:%8d:%5d%5d"+
		":%5d%5d%5d%5d%5d%5d%5d:%8dms (%8.1fms)\n"
		,s,d	,sp[ 0],sp[ 1],sp[ 2],sp[ 3]
			,sp[ 4],sp[ 5],sp[ 6],sp[ 7] ,sp[ 8]
			,sp[ 9],sp[10],sp[14],(double)sp[14]/(double)nort);
	  }else{
	    if(f	!=0)	f++;
      } }}}
      if(noe==0){
	printf("\nSet STATISINFO=1 in the .ini file when using this function.\n\n");
      }else	printf("\n");
      rtn=0;
    }

    if(rtn<(-999) || ((-900)<rtn && rtn<0)){
      printf("(E) I/O Error (%d)\n"	,rtn);
  } }

  // If you want to terminate (call _TDSCommClose() or _TDSUDrvClose()) like
  // this AP, you don't need the following "Separate request" processing.
  //  AP ̗lɁÂ܂܏I (_TDSCommClose()  _TDSUDrvClose()
  //  Call ) ̂ł΁Aȉ "Separate v" ͕KvȂB
  if(SECS_MODE	!=0){			// In case of HSMS, Shutdown process
					// HSMS ڑ̏ꍇAؒf
    if(FuncType==0)	rtn=_TDJVSCommStatus(Fd,0);
    else		rtn=_TDJVSUDrvStatus(Fd,0);
    if(rtn	==3){
    // Deselect request is not performed. (Of course you may go. However SEMI
    // claims that HSMS-SS does not perform Deselect request.)
    // Deselect request ͍sȂB (sĂ悢BSEMI ł HSMS-SS
    // ɂāADeselect request ͍sȂAƂĂB)
    //if(FuncType==0)	rtn=_TDJVSCommSend(Fd,0x0800,0,0,0,msg,0,hd);
    //else		rtn=_TDJVSUDrvSend(Fd,0x0800,0,0,0,msg,0,hd);
    //if(rtn							< 0) break L0;
    //printf("(E) Deselected\n");
      if(FuncType==0)	rtn=_TDJVSCommSend(Fd,0x0900,0,0,0,msg,0,hd);
      else		rtn=_TDJVSUDrvSend(Fd,0x0900,0,0,0,msg,0,hd);
      if(rtn							< 0) break L0;
      printf("(E) Separated\n");
  } }
  }	// L0:

  Break= 1;
  if(Fd> 0){
    if(FuncType==0)	_TDJVSCommClose(Fd,0);
    else		_TDJVSUDrvClose(Fd,0);
  } else		printf("(E) Error (%d)\n"	,Fd);
  if(Md> 0)		_TDJVSMDMssgTerminate(Md,0);
}

}
